/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

import org.eclipse.paho.client.mqttv3.MqttAsyncClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;

import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

/**
 * Utility class for publishing MQTT messages during tests.
 * Useful for race condition testing and pre-publishing scenarios.
 */
public class MQTT3MessagePublisher {

  private static final String DEFAULT_BROKER_URL = "tcp://127.0.0.1:1883";
  private static final String DEFAULT_USERNAME = "mosquitto";
  private static final String DEFAULT_PASSWORD = "mosquitto";
  private static final int DEFAULT_TIMEOUT = 10;

  /**
   * Publishes a single message to the specified topic.
   *
   * @param brokerUrl The MQTT broker URL (e.g., "tcp://127.0.0.1:1883")
   * @param topic     The topic to publish to
   * @param message   The message content
   * @param qos       Quality of Service (0, 1, or 2)
   * @param retained  Whether the message should be retained
   * @throws Exception If publishing fails
   */
  public static void publishMessage(String brokerUrl, String topic, String message, int qos, boolean retained)
      throws Exception {
    publishMessage(brokerUrl, topic, message, qos, retained, DEFAULT_USERNAME, DEFAULT_PASSWORD);
  }

  /**
   * Publishes a single message with custom credentials.
   */
  public static void publishMessage(String brokerUrl, String topic, String message, int qos, boolean retained,
                                    String username, String password)
      throws Exception {

    String clientId = "TestPublisher-" + UUID.randomUUID().toString();
    MqttAsyncClient client = null;

    try {
      client = new MqttAsyncClient(brokerUrl, clientId, new MemoryPersistence());

      MqttConnectOptions options = new MqttConnectOptions();
      options.setUserName(username);
      if (password != null) {
        options.setPassword(password.toCharArray());
      }
      options.setConnectionTimeout(DEFAULT_TIMEOUT);
      options.setCleanSession(true);

      // Connect synchronously
      CountDownLatch connectLatch = new CountDownLatch(1);
      final Exception[] connectException = new Exception[1];

      client.connect(options, null, new org.eclipse.paho.client.mqttv3.IMqttActionListener() {

        @Override
        public void onSuccess(org.eclipse.paho.client.mqttv3.IMqttToken token) {
          connectLatch.countDown();
        }

        @Override
        public void onFailure(org.eclipse.paho.client.mqttv3.IMqttToken token, Throwable exception) {
          connectException[0] = new Exception("Connection failed", exception);
          connectLatch.countDown();
        }
      });

      if (!connectLatch.await(DEFAULT_TIMEOUT, TimeUnit.SECONDS)) {
        throw new Exception("Connection timeout");
      }

      if (connectException[0] != null) {
        throw connectException[0];
      }

      // Publish message
      MqttMessage mqttMessage = new MqttMessage(message.getBytes());
      mqttMessage.setQos(qos);
      mqttMessage.setRetained(retained);

      CountDownLatch publishLatch = new CountDownLatch(1);
      final Exception[] publishException = new Exception[1];

      client.publish(topic, mqttMessage, null, new org.eclipse.paho.client.mqttv3.IMqttActionListener() {

        @Override
        public void onSuccess(org.eclipse.paho.client.mqttv3.IMqttToken token) {
          System.out.println("Successfully published message to topic: " + topic);
          publishLatch.countDown();
        }

        @Override
        public void onFailure(org.eclipse.paho.client.mqttv3.IMqttToken token, Throwable exception) {
          publishException[0] = new Exception("Publish failed", exception);
          publishLatch.countDown();
        }
      });

      if (!publishLatch.await(DEFAULT_TIMEOUT, TimeUnit.SECONDS)) {
        throw new Exception("Publish timeout");
      }

      if (publishException[0] != null) {
        throw publishException[0];
      }

    } finally {
      if (client != null && client.isConnected()) {
        client.disconnect();
        client.close();
      }
    }
  }

  /**
   * Publishes a message with specific clientId and session settings for persistence testing.
   *
   * @param brokerUrl    The MQTT broker URL
   * @param topic        The topic to publish to
   * @param message      The message content
   * @param qos          Quality of Service (0, 1, or 2)
   * @param retained     Whether the message should be retained
   * @param clientId     Specific client ID to use
   * @param cleanSession Whether to use clean session
   * @param username     MQTT username
   * @param password     MQTT password
   * @throws Exception If publishing fails
   */
  public static void publishMessageWithPersistence(String brokerUrl, String topic, String message, int qos,
                                                   boolean retained, String clientId, boolean cleanSession,
                                                   String username, String password)
      throws Exception {

    MqttAsyncClient client = null;

    try {
      client = new MqttAsyncClient(brokerUrl, clientId, new MemoryPersistence());

      MqttConnectOptions options = new MqttConnectOptions();
      options.setUserName(username);
      if (password != null) {
        options.setPassword(password.toCharArray());
      }
      options.setConnectionTimeout(DEFAULT_TIMEOUT);
      options.setCleanSession(cleanSession);

      // Connect synchronously
      CountDownLatch connectLatch = new CountDownLatch(1);
      final Exception[] connectException = new Exception[1];

      client.connect(options, null, new org.eclipse.paho.client.mqttv3.IMqttActionListener() {

        @Override
        public void onSuccess(org.eclipse.paho.client.mqttv3.IMqttToken token) {
          connectLatch.countDown();
        }

        @Override
        public void onFailure(org.eclipse.paho.client.mqttv3.IMqttToken token, Throwable exception) {
          connectException[0] = new Exception("Connection failed", exception);
          connectLatch.countDown();
        }
      });

      if (!connectLatch.await(DEFAULT_TIMEOUT, TimeUnit.SECONDS)) {
        throw new Exception("Connection timeout");
      }

      if (connectException[0] != null) {
        throw connectException[0];
      }

      // Publish message
      MqttMessage mqttMessage = new MqttMessage(message.getBytes());
      mqttMessage.setQos(qos);
      mqttMessage.setRetained(retained);

      CountDownLatch publishLatch = new CountDownLatch(1);
      final Exception[] publishException = new Exception[1];

      client.publish(topic, mqttMessage, null, new org.eclipse.paho.client.mqttv3.IMqttActionListener() {

        @Override
        public void onSuccess(org.eclipse.paho.client.mqttv3.IMqttToken token) {
          System.out.println("Successfully published persistent message to topic: " + topic + " with clientId: " + clientId);
          publishLatch.countDown();
        }

        @Override
        public void onFailure(org.eclipse.paho.client.mqttv3.IMqttToken token, Throwable exception) {
          publishException[0] = new Exception("Publish failed", exception);
          publishLatch.countDown();
        }
      });

      if (!publishLatch.await(DEFAULT_TIMEOUT, TimeUnit.SECONDS)) {
        throw new Exception("Publish timeout");
      }

      if (publishException[0] != null) {
        throw publishException[0];
      }

    } finally {
      if (client != null && client.isConnected()) {
        client.disconnect();
        client.close();
      }
    }
  }

  /**
   * Publishes multiple messages to the same topic with a delay between each.
   */
  public static void publishBurstMessages(String brokerUrl, String topic, String messagePrefix,
                                          int messageCount, int qos, boolean retained, long delayMs)
      throws Exception {
    for (int i = 1; i <= messageCount; i++) {
      String message = messagePrefix + "-" + i + "-" + System.currentTimeMillis();
      publishMessage(brokerUrl, topic, message, qos, retained);

      if (delayMs > 0 && i < messageCount) {
        Thread.sleep(delayMs);
      }
    }
    System.out.println("Published " + messageCount + " messages to topic: " + topic);
  }

  /**
   * Publishes retained messages for race condition testing.
   */
  public static void publishRetainedMessages(String brokerUrl, String topicPrefix, int messageCount) throws Exception {
    for (int i = 1; i <= messageCount; i++) {
      String topic = topicPrefix + "/" + i;
      String message = "RetainedMessage-" + i + "-" + System.currentTimeMillis();
      publishMessage(brokerUrl, topic, message, 1, true); // QoS 1, retained true
    }
    System.out.println("Published " + messageCount + " retained messages with topic prefix: " + topicPrefix);
  }

  /**
   * Convenience method using default broker URL (for use with ${mosquitto.port1}).
   */
  public static void publishToDefaultBroker(String topic, String message, boolean retained) throws Exception {
    // Note: In real tests, this would use the dynamic port from Maven
    String brokerUrl = System.getProperty("mqtt.test.broker.url", DEFAULT_BROKER_URL);
    publishMessage(brokerUrl, topic, message, 1, retained);
  }

  /**
   * Publishes a rapid burst of messages for stress testing.
   */
  public static void publishRapidBurst(String brokerUrl, String topic, int messageCount) throws Exception {
    publishBurstMessages(brokerUrl, topic, "RapidBurst", messageCount, 1, false, 0);
  }
}
