/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

import org.eclipse.paho.client.mqttv3.MqttAsyncClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

/**
 * Utility class for creating durable MQTT subscriptions during tests.
 * This is used to establish persistent sessions that survive client disconnections.
 */
public class MQTT3MessageSubscriber {

  private static final int DEFAULT_TIMEOUT = 10;

  /**
   * Creates a durable subscription by connecting, subscribing, and then disconnecting
   * while maintaining the session (cleanSession=false).
   *
   * @param brokerUrl The MQTT broker URL
   * @param topic     The topic to subscribe to
   * @param clientId  The client ID for the durable session
   * @param username  MQTT username
   * @param password  MQTT password
   * @throws Exception If the operation fails
   */
  public static void createDurableSubscription(String brokerUrl, String topic, String clientId,
                                               String username, String password)
      throws Exception {

    MqttAsyncClient client = null;

    try {
      client = new MqttAsyncClient(brokerUrl, clientId, new MemoryPersistence());

      MqttConnectOptions options = new MqttConnectOptions();
      options.setUserName(username);
      if (password != null) {
        options.setPassword(password.toCharArray());
      }
      options.setConnectionTimeout(DEFAULT_TIMEOUT);
      options.setCleanSession(false); // Keep session for message persistence

      // Connect synchronously
      CountDownLatch connectLatch = new CountDownLatch(1);
      final Exception[] connectException = new Exception[1];

      client.connect(options, null, new org.eclipse.paho.client.mqttv3.IMqttActionListener() {

        @Override
        public void onSuccess(org.eclipse.paho.client.mqttv3.IMqttToken token) {
          connectLatch.countDown();
        }

        @Override
        public void onFailure(org.eclipse.paho.client.mqttv3.IMqttToken token, Throwable exception) {
          connectException[0] = new Exception("Connection failed", exception);
          connectLatch.countDown();
        }
      });

      if (!connectLatch.await(DEFAULT_TIMEOUT, TimeUnit.SECONDS)) {
        throw new Exception("Connection timeout");
      }

      if (connectException[0] != null) {
        throw connectException[0];
      }

      // Subscribe to topic with QoS 1
      CountDownLatch subscribeLatch = new CountDownLatch(1);
      final Exception[] subscribeException = new Exception[1];

      client.subscribe(topic, 1, null, new org.eclipse.paho.client.mqttv3.IMqttActionListener() {

        @Override
        public void onSuccess(org.eclipse.paho.client.mqttv3.IMqttToken token) {
          System.out.println("Successfully created durable subscription to topic: " + topic + " with clientId: " + clientId);
          subscribeLatch.countDown();
        }

        @Override
        public void onFailure(org.eclipse.paho.client.mqttv3.IMqttToken token, Throwable exception) {
          subscribeException[0] = new Exception("Subscribe failed", exception);
          subscribeLatch.countDown();
        }
      });

      if (!subscribeLatch.await(DEFAULT_TIMEOUT, TimeUnit.SECONDS)) {
        throw new Exception("Subscribe timeout");
      }

      if (subscribeException[0] != null) {
        throw subscribeException[0];
      }

      // Wait a moment to ensure subscription is registered
      Thread.sleep(1000);

      System.out.println("Durable subscription established. Disconnecting client to simulate offline scenario.");

    } finally {
      if (client != null && client.isConnected()) {
        // Disconnect but keep session (cleanSession was false)
        client.disconnect();
        client.close();
      }
    }
  }
}
