/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.connection.provider;

import com.mulesoft.connectors.mqtt3.internal.connection.MQTT3Connection;
import com.mulesoft.connectors.mqtt3.internal.utils.MQTT3ConnectionProviderConcrete;
import org.junit.Test;

import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionValidationResult;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class MQTT3ConnectionProviderTest {

  @Test
  public void equalsTest() {
    MQTT3ConnectionProvider connectionProviderFormConnection = new MQTT3FormConnectionProvider();
    assertTrue(connectionProviderFormConnection.equals(connectionProviderFormConnection));
    MQTT3ConnectionProvider connectionProviderFailOverConnection = new MQTT3FailOverConnectionProvider();
    assertTrue(connectionProviderFailOverConnection.equals(connectionProviderFailOverConnection));
    MQTT3ConnectionProvider connectionProviderUrl = new MQTT3UrlConnectionProvider();
    assertTrue(connectionProviderUrl.equals(connectionProviderUrl));
    String anotherObject = "anotherObject";
    assertFalse(connectionProviderFormConnection.equals(anotherObject));
    assertFalse(connectionProviderFailOverConnection.equals(anotherObject));
    assertFalse(connectionProviderUrl.equals(anotherObject));
  }

  @Test
  public void validateConnectionTestsWithConnectionNull() {
    MQTT3ConnectionProvider connectionProviderFormConnection = new MQTT3FormConnectionProvider();
    ConnectionValidationResult result = connectionProviderFormConnection.validate(null);
    assertNotNull(result);
    assertFalse(result.isValid());
    assertEquals("Disconnected", result.getMessage());
  }

  @Test
  public void validateConnectionTestsWithValidConnection() throws Exception {
    MQTT3Connection mockMQTT3Connection = mock(MQTT3Connection.class);
    when(mockMQTT3Connection.testConnectivity()).thenReturn(true);
    MQTT3ConnectionProvider connectionProviderFormConnection = new MQTT3FormConnectionProvider();
    ConnectionValidationResult result = connectionProviderFormConnection.validate(mockMQTT3Connection);
    assertNotNull(result);
    assertTrue(result.isValid());
  }

  @Test
  public void validateConnectionTestsWithInvalidConnection() throws Exception {
    MQTT3Connection mockMQTT3Connection = mock(MQTT3Connection.class);
    when(mockMQTT3Connection.testConnectivity()).thenReturn(false);
    MQTT3ConnectionProvider connectionProviderFormConnection = new MQTT3FormConnectionProvider();
    ConnectionValidationResult result = connectionProviderFormConnection.validate(mockMQTT3Connection);
    assertNotNull(result);
    assertFalse(result.isValid());
    assertEquals("Disconnected", result.getMessage());
  }

  @Test
  public void hashTest() {
    MQTT3ConnectionProvider connectionProviderFormConnection = new MQTT3FormConnectionProvider();
    MQTT3ConnectionProvider connectionProviderFailOverConnection = new MQTT3FailOverConnectionProvider();
    MQTT3ConnectionProvider connectionProviderUrl = new MQTT3UrlConnectionProvider();
    MQTT3ConnectionProviderConcrete connectionProviderConcreteForTest = new MQTT3ConnectionProviderConcrete();
    assertEquals(connectionProviderFormConnection.hashCode(), connectionProviderFormConnection.hashCode());
    assertEquals(connectionProviderFailOverConnection.hashCode(), connectionProviderFailOverConnection.hashCode());
    assertEquals(connectionProviderUrl.hashCode(), connectionProviderUrl.hashCode());
    assertEquals(connectionProviderConcreteForTest.hashCode(), connectionProviderConcreteForTest.hashCode());
  }

  @Test
  public void abstractConnectionProviderTest() {
    MQTT3ConnectionProviderConcrete provider = new MQTT3ConnectionProviderConcrete();
    provider.setUserName("user");
    provider.setPassword("password");
    assertTrue(provider.equals(provider));
    String anotherObject = "anotherObject";
    assertFalse(provider.equals(anotherObject));

    MQTT3ConnectionProviderConcrete anotherProviderWithSameConfig = new MQTT3ConnectionProviderConcrete();
    anotherProviderWithSameConfig.setUserName("user");
    anotherProviderWithSameConfig.setPassword("password");
    assertTrue(provider.equals(anotherProviderWithSameConfig));

    MQTT3ConnectionProviderConcrete anotherProvider = new MQTT3ConnectionProviderConcrete();
    anotherProvider.setUserName("user123");
    anotherProvider.setPassword("password123");
    assertFalse(provider.equals(anotherProvider));
  }

  @Test
  public void testValidateWithExceptionInTestConnectivity() {
    MQTT3Connection mockMQTT3Connection = mock(MQTT3Connection.class);
    try {
      when(mockMQTT3Connection.testConnectivity()).thenThrow(new RuntimeException("Test connectivity failed"));
    } catch (ConnectionException e) {
      throw new RuntimeException(e);
    }
    MQTT3ConnectionProvider connectionProviderFormConnection = new MQTT3FormConnectionProvider();
    ConnectionValidationResult result = connectionProviderFormConnection.validate(mockMQTT3Connection);
    assertNotNull(result);
    assertFalse(result.isValid());
    assertEquals("Disconnected", result.getMessage());
  }
}
