/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity;

import static java.nio.file.Files.createDirectory;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.mule.connectivity.restconnect.api.Parser;
import org.mule.connectivity.restconnect.api.SpecFormat;

import com.beust.jcommander.Parameter;
import com.beust.jcommander.converters.FileConverter;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

public abstract class AbstractRestConnectCommand {

  @Parameter(names = "-apiSpec", description = "Api Specification", required = true, converter = FileConverter.class)
  private File apiSpec;

  @Parameter(names = "-outputDir")
  private String outputDir;

  @Parameter(names = "-specFormat", description = "API Specification format -- Possible values: RAML, JsonOAS, YamlOAS")
  private String specFormat;

  @Parameter(names = "-parser", description = "Parser -- Possible values: RamlParser, AMF")
  private String parser;

  @Parameter(names = "-skipValidation", description = "Skip Validation -- Possible values: true, false")
  private String skipValidation;

  public File getApiSpec() {
    return apiSpec;
  }

  public Path getOutputDir() throws IOException {
    Path output;

    if (isNotBlank(outputDir)) {
      if (Files.exists(Paths.get(outputDir))) {
        output = Paths.get(outputDir);
      } else {
        output = createDirectory(Paths.get(outputDir));
      }
    } else {
      output = Paths.get(".").toAbsolutePath().normalize();
    }

    return output;
  }

  public SpecFormat getSpecFormat() {
    if (isBlank(specFormat)) {
      return SpecFormat.RAML;
    }
    return SpecFormat.getFromString(specFormat);
  }

  public Parser getParser() {
    if (isBlank(parser)) {
      return Parser.AMF;
    }
    return Parser.getFromString(parser);
  }

  public boolean getSkipValidation() {
    if (isBlank(skipValidation)) {
      return Boolean.parseBoolean(skipValidation);
    }
    return false;
  }
}
