/**
 * Mule Google Api Commons
 *
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package com.google.gdata.util.common.base;

import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.VisibleForTesting;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nullable;

/**
 * Helper functions that can operate on any {@code Object}.
 *
 * 
 */
@GwtCompatible
public final class Objects {
  private Objects() {}

  /**
   * Determines whether two possibly-null objects are equal. Returns:
   *
   * <ul>
   * <li>{@code true} if {@code a} and {@code b} are both null.
   * <li>{@code true} if {@code a} and {@code b} are both non-null and they are
   *     equal according to {@link Object#equals(Object)}.
   * <li>{@code false} in all other situations.
   * </ul>
   *
   * <p>This assumes that any non-null objects passed to this function conform
   * to the {@code equals()} contract.
   */
  public static boolean equal(Object a, Object b) {
    return a == b || (a != null && a.equals(b));
  }

  /**
   * Generates a hash code for multiple values. The hash code is generated by
   * calling {@link Arrays#hashCode(Object[])}.
   *
   * <p>This is useful for implementing {@link Object#hashCode()}. For example,
   * in an object that has three properties, {@code x}, {@code y}, and
   * {@code z}, one could write:
   * <pre>
   * public int hashCode() {
   *   return Objects.hashCode(getX(), getY(), getZ());
   * }</pre>
   *
   * <b>Warning</b>: When a single object is supplied, the returned hash code
   * does not equal the hash code of that object.
   */
  public static int hashCode(Object... objects) {
    return Arrays.hashCode(objects);
  }
  
  /**
   * Creates an instance of {@link ToStringHelper}.
   * 
   * <p>This is helpful for implementing {@link Object#toString()}. For 
   * example, in an object that contains two member variables, {@code x}, 
   * and {@code y}, one could write:<pre>   <tt> 
   *   public class ClassName {
   *     public String toString() {
   *       return Objects.toStringHelper(this)
   *           .add("x", x)
   *           .add("y", y)
   *           .toString();
   *     }
   *   }</tt>
   * </pre>
   * 
   * Assuming the values of {@code x} and {@code y} are 1 and 2, 
   * this code snippet returns the string <tt>"ClassName{x=1, y=2}"</tt>. 
   */
  public static ToStringHelper toStringHelper(Object object) {
    return new ToStringHelper(object);
  }

  /**
   * Checks that the specified object is not {@code null}.
   *
   * @param obj the object to check for nullness
   * @return {@code obj} if not null.
   * @throws NullPointerException if {@code obj} is null.
   */
  // NOTE: not present in the external version of this library
  public static <T> T nonNull(T obj) {
    if (obj == null) {
      throw new NullPointerException();
    }
    return obj;
  }

  /**
   * Checks that the specified object is not {@code null}.
   *
   * @param obj the object to check for nullness
   * @param message exception message used in the event that a {@code
   *     NullPointerException} is thrown
   * @return {@code obj} if not null
   * @throws NullPointerException if {@code obj} is null
   */
  // NOTE: not present in the external version of this library
  public static <T> T nonNull(T obj, String message) {
    if (obj == null) {
      throw new NullPointerException(message);
    }
    return obj;
  }

  /**
   * Returns the first of two given parameters that is not {@code null}, if
   * either is, or otherwise throws a {@link NullPointerException}.
   *
   * @return {@code first} if {@code first} is not {@code null}, or
   *     {@code second} if {@code first} is {@code null} and {@code second} is
   *     not {@code null}
   * @throws NullPointerException if both {@code first} and {@code second} were
   *     {@code null}
   */
  // NOTE: not present in the external version of this library
  public static <T> T firstNonNull(T first, T second) {
    return first != null ? first : Preconditions.checkNotNull(second);
  }

  /**
   * Determines if two objects are equal as determined by
   * {@link Object#equals(Object)}, or "deeply equal" if both are arrays.
   *
   * <p>If both objects are null, {@code true} is returned. If only one is null,
   * {@code false} is returned. If both objects are arrays, the corresponding
   * {@link Arrays#deepEquals(Object[], Object[])}, or
   * {@link Arrays#equals(int[], int[])} or the like are called to determine
   * equality. Otherwise, {@code a.equals(b)} is returned.
   *
   * <p>Note that this method does not "deeply" compare the fields of the
   * objects.
   */
  // NOTE: not present in the external version of this library
  public static boolean deepEquals(Object a, Object b) {
    if (a == b) {
      return true;
    }
    if (a == null || b == null) {
      return false;
    }

    Class<?> type1 = a.getClass();
    Class<?> type2 = b.getClass();
    if (!(type1.isArray() && type2.isArray())) {
      return a.equals(b);
    }
    if (a instanceof Object[] && b instanceof Object[]) {
      return Arrays.deepEquals((Object[]) a, (Object[]) b);
    }
    if (type1 != type2) {
      return false;
    }
    if (a instanceof boolean[]) {
      return Arrays.equals((boolean[]) a, (boolean[]) b);
    }
    if (a instanceof char[]) {
      return Arrays.equals((char[]) a, (char[]) b);
    }
    if (a instanceof byte[]) {
      return Arrays.equals((byte[]) a, (byte[]) b);
    }
    if (a instanceof short[]) {
      return Arrays.equals((short[]) a, (short[]) b);
    }
    if (a instanceof int[]) {
      return Arrays.equals((int[]) a, (int[]) b);
    }
    if (a instanceof long[]) {
      return Arrays.equals((long[]) a, (long[]) b);
    }
    if (a instanceof float[]) {
      return Arrays.equals((float[]) a, (float[]) b);
    }
    if (a instanceof double[]) {
      return Arrays.equals((double[]) a, (double[]) b);
    }
    throw new AssertionError();
  }

  /**
   * Determines the hash code of an object, returning a hash code based on the
   * array's "deep contents" if the object is an array.
   *
   * <p>If {@code obj} is null, 0 is returned. If {@code obj} is an array, the
   * corresponding {@link Arrays#deepHashCode(Object[])}, or
   * {@link Arrays#hashCode(int[])} or the like is used to calculate the hash
   * code. If {@code obj} isn't null or an array, {@code obj.hashCode()} is
   * returned.
   */
  // NOTE: not present in the external version of this library
  public static int deepHashCode(Object obj) {
    if (obj == null) {
      return 0;
    }
    if (!obj.getClass().isArray()) {
      return obj.hashCode();
    }
    if (obj instanceof Object[]) {
      return Arrays.deepHashCode((Object[]) obj);
    }
    if (obj instanceof boolean[]) {
      return Arrays.hashCode((boolean[]) obj);
    }
    if (obj instanceof char[]) {
      return Arrays.hashCode((char[]) obj);
    }
    if (obj instanceof byte[]) {
      return Arrays.hashCode((byte[]) obj);
    }
    if (obj instanceof short[]) {
      return Arrays.hashCode((short[]) obj);
    }
    if (obj instanceof int[]) {
      return Arrays.hashCode((int[]) obj);
    }
    if (obj instanceof long[]) {
      return Arrays.hashCode((long[]) obj);
    }
    if (obj instanceof float[]) {
      return Arrays.hashCode((float[]) obj);
    }
    if (obj instanceof double[]) {
      return Arrays.hashCode((double[]) obj);
    }
    throw new AssertionError();
  }

  /**
   * Determines the string representation of an object, or the "deep contents
   * of the array if the {@code obj} is an array.
   *
   * <p>If {@code obj} is null, {@code "null"} is returned; if {@code obj} is an
   * array, the corresponding {@link Arrays#deepToString(Object[])},
   * {@link Arrays#toString(int[])} or the like is used to generate the string.
   * If {@code obj} isn't null or an array, {@code obj.toString()} is returned.
   */
  // NOTE: not present in the external version of this library
  public static String deepToString(Object obj) {
    if (obj == null) {
      return String.valueOf(obj);
    }
    if (!obj.getClass().isArray()) {
      return obj.toString();
    }
    if (obj instanceof Object[]) {
      return Arrays.deepToString((Object[]) obj);
    }
    if (obj instanceof boolean[]) {
      return Arrays.toString((boolean[]) obj);
    }
    if (obj instanceof char[]) {
      return Arrays.toString((char[]) obj);
    }
    if (obj instanceof byte[]) {
      return Arrays.toString((byte[]) obj);
    }
    if (obj instanceof short[]) {
      return Arrays.toString((short[]) obj);
    }
    if (obj instanceof int[]) {
      return Arrays.toString((int[]) obj);
    }
    if (obj instanceof long[]) {
      return Arrays.toString((long[]) obj);
    }
    if (obj instanceof float[]) {
      return Arrays.toString((float[]) obj);
    }
    if (obj instanceof double[]) {
      return Arrays.toString((double[]) obj);
    }
    throw new AssertionError();
  }
  
  /**
   * Helper class for generating consistent toString across ads/api pojos.<p/>
   * 
   * This class is made to support {@link Objects#toStringHelper(Object)}.<p/>
   * 
   * @see Objects#toStringHelper(Object)
   * 
   */
  public static class ToStringHelper {
    
    private final List<String> fieldString = new ArrayList<String>();
    private final Object instance;

    /**
     * Use {@link Objects#toStringHelper(Object)} to create an instance.
     */
    private ToStringHelper(Object instance) {
      this.instance = Preconditions.checkNotNull(instance);
    }

    /**
     * Adds a name/value pair to the formatted output in {@code name=value}
     * format. If {@code value} is {@code null}, the string {@code "null"} 
     * is used.
     */
    public ToStringHelper add(String name, Object value) {
      return addValue(Preconditions.checkNotNull(name) + "=" + value);
    }
    
    /**
     * Adds a value to the formatted output in {@code value} format.<p/>
     * 
     * It is strongly encouraged to use {@link #add(String, Object)} instead and
     * give value a readable name. 
     */
    public ToStringHelper addValue(Object value) {
      fieldString.add(String.valueOf(value));
      return this;
    }

    private static final Joiner JOINER = Joiner.on(", ");

    /**
     * Returns the formatted string.
     */
    @Override public String toString() {
      StringBuilder builder = new StringBuilder(100)
          .append(simpleName(instance.getClass()))
          .append('{');
      return JOINER.appendTo(builder, fieldString)
          .append('}')
          .toString();
    }

    /**
     * {@link Class#getSimpleName()} is not GWT compatible yet, so we
     * provide our own implementation.
     */
    @VisibleForTesting
    static String simpleName(Class<?> clazz) {
      String name = clazz.getName();

      // we want the name of the inner class all by its lonesome
      int start = name.lastIndexOf('$');

      // if this isn't an inner class, just find the start of the
      // top level class name.
      if (start == -1) {
        start = name.lastIndexOf('.');
      }
      return name.substring(start + 1);
    }
  }
}
