/**
 * Mule Google Api Commons
 *
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */


package com.google.gdata.data.batch;

import com.google.gdata.util.common.xml.XmlWriter;
import com.google.gdata.client.CoreErrorDomain;
import com.google.gdata.data.Extension;
import com.google.gdata.data.ExtensionDescription;
import com.google.gdata.data.ExtensionPoint;
import com.google.gdata.data.ExtensionProfile;
import com.google.gdata.util.Namespaces;
import com.google.gdata.util.ParseException;
import com.google.gdata.util.XmlParser;

import org.xml.sax.Attributes;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

/**
 * Entry and Feed extension that defines the batch operation
 * to apply, defined by its type ({@link BatchOperationType}).
 *
 * 
 */
public class BatchOperation extends ExtensionPoint implements Extension {

  private BatchOperationType type;

  /** Creates a new BatchOperation object. */
  public BatchOperation(BatchOperationType type) {
    this.type = type;
  }

  /** Creates a BatchOperation without type. */
  public BatchOperation() {
    
  }

  /** Returns the suggested extension description. */
  public static ExtensionDescription getDefaultDescription() {
    ExtensionDescription desc = new ExtensionDescription();
    desc.setExtensionClass(BatchOperation.class);
    desc.setNamespace(Namespaces.batchNs);
    desc.setLocalName("operation");
    desc.setRepeatable(false);
    return desc;
  }
  
  /**
   * Gets the operation type ({@link BatchOperationType#DELETE}, 
   * {@link BatchOperationType#INSERT} or {@link BatchOperationType#UPDATE}.
   */
  public BatchOperationType getType() {
    return type;
  }
  
  /** Sets the operation type. */
  public void setType(BatchOperationType type) {
    this.type = type;
  }
  
  @Override
  public void generate(XmlWriter w, ExtensionProfile extProfile)
      throws IOException {

    List<XmlWriter.Attribute> attrs = null;
    if (type != null) {
      attrs = Collections.singletonList(
          new XmlWriter.Attribute("type", this.type.getName()));
    }
    generateStartElement(w, Namespaces.batchNs, "operation", attrs, null);
    
    // Generate extension elements
    generateExtensions(w, extProfile);
    
    w.endElement(Namespaces.batchNs, "operation");
  }

  @Override
  public XmlParser.ElementHandler getHandler(ExtensionProfile extProfile,
                                             String namespace, String localName,
                                             Attributes attrs)
      throws ParseException {
    return new BatchOperationHandler(extProfile, attrs);
  }
  
  private class BatchOperationHandler extends ExtensionPoint.ExtensionHandler {

    public BatchOperationHandler(ExtensionProfile profile, Attributes attrs)
        throws ParseException {
      super(profile, BatchOperation.class);
      String operationType = attrs.getValue("type");
      BatchOperationType op = BatchOperationType.forName(operationType);
      if (op == null) {
        ParseException pe = new ParseException(
            CoreErrorDomain.ERR.invalidBatchOperationType);
        pe.setInternalReason("Invalid type for batch:operation: '" +
            operationType + "'");
        throw pe;
      }
      type = op;
    }
  }
}
