/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */
package com.mulesoft.modules.agent.conductor.internal.prompt;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.core.api.util.StringUtils.isBlank;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.mule.runtime.core.api.util.IOUtils;

import com.mulesoft.modules.agent.conductor.internal.tool.Tool;

import java.util.Collection;
import java.util.List;

public class PromptBuilder {

  private static final String PROMPT_TEMPLATE;

  static {
    try {
      PROMPT_TEMPLATE = IOUtils.toString(PromptBuilder.class.getClassLoader()
          .getResourceAsStream("agent-loop/prompt-template.txt"));
    } catch (Exception e) {
      throw new MuleRuntimeException(createStaticMessage("Failed to load prompt template"), e);
    }
  }

  private final ExpressionManager expressionManager;

  private String userPrompt;
  private String userInstructions;
  private Collection<Tool> tools = List.of();
  private String groundings;
  private String conversationHistory;
  private Integer currentLoopIteration = 1;
  private Integer maxLoops;

  private String formattedTools;

  public PromptBuilder(ExpressionManager expressionManager) {
    this.expressionManager = expressionManager;
  }

  /**
   * Sets the user prompt.
   *
   * @param userPrompt the user prompt to set
   * @return this PromptBuilder instance for method chaining
   * @throws IllegalArgumentException if userPrompt is null or empty
   */
  public PromptBuilder setUserPrompt(String userPrompt) {
    if (isBlank(userPrompt)) {
      throw new IllegalArgumentException("User prompt cannot be null or empty");
    }
    this.userPrompt = userPrompt;
    return this;
  }

  /**
   * Sets the available tools
   *
   * @param tools the available tools
   * @return this PromptBuilder instance for method chaining
   */
  public PromptBuilder setTools(Collection<Tool> tools) {
    this.tools = tools != null ? tools : List.of();
    this.formattedTools = null;
    return this;
  }

  /**
   * Sets the conversation history.
   *
   * @param conversationHistory the conversation history to set
   * @return this PromptBuilder instance for method chaining
   * @throws IllegalArgumentException if conversationHistory is null or empty
   */
  public PromptBuilder setConversationHistory(String conversationHistory) {
    this.conversationHistory = conversationHistory;
    return this;
  }

  /**
   * Sets the current loop iteration.
   *
   * @param currentLoopIteration the current loop iteration to set
   * @return this PromptBuilder instance for method chaining
   * @throws IllegalArgumentException if currentLoopIteration is null or negative
   */
  public PromptBuilder setCurrentLoopIteration(Integer currentLoopIteration) {
    if (currentLoopIteration == null) {
      throw new IllegalArgumentException("Current loop iteration cannot be null");
    }
    if (currentLoopIteration < 1) {
      throw new IllegalArgumentException("Current loop iteration cannot be lower than one");
    }
    this.currentLoopIteration = currentLoopIteration;
    return this;
  }

  /**
   * Sets the maximum number of loops.
   *
   * @param maxLoops the maximum number of loops to set
   * @return this PromptBuilder instance for method chaining
   * @throws IllegalArgumentException if maxLoops is null or not positive
   */
  public PromptBuilder setMaxLoops(Integer maxLoops) {
    if (maxLoops == null) {
      throw new IllegalArgumentException("Max loops cannot be null");
    }
    if (maxLoops < 1) {
      throw new IllegalArgumentException("Max loops cannot be lower than 1");
    }
    this.maxLoops = maxLoops;
    return this;
  }

  public PromptBuilder setGroundings(String groundings) {
    this.groundings = groundings;
    return this;
  }

  public PromptBuilder setUserInstructions(String userInstructions) {
    this.userInstructions = userInstructions;
    return this;
  }

  public String build() {
    // TODO: Replace with DW for perf
    return PROMPT_TEMPLATE
        .replace("{{userPrompt}}", userPrompt)
        .replace("{{instructions}}", nullSafe(userInstructions))
        .replace("{{tools}}", getFormattedTools())
        .replace("{{conversationHistory}}", nullSafe(conversationHistory))
        .replace("{{currentLoopIteration}}", String.valueOf(currentLoopIteration))
        .replace("{{groundings}}", nullSafe(groundings))
        .replace("{{maxLoops}}", String.valueOf(maxLoops));
  }

  private String nullSafe(String value) {
    return isBlank(value) ? "None" : value;
  }

  private String getFormattedTools() {
    if (formattedTools == null) {
      StringBuilder builder = new StringBuilder();
      if (tools.isEmpty()) {
        formattedTools = "No Tools available";
      } else {
        tools.forEach(tool -> {
          builder.append("\n- **").append(tool.getId()).append("**:\n")
              .append("Description:\n```\n").append(tool.getDescription()).append("\n```\n")
              .append("\nInput:\n```\n").append(tool.getInput()).append("\n```\n")
              .append("\nOutput:\n```\n").append(tool.getOutput()).append("\n```\n\n");
        });
        formattedTools = builder.toString();
      }
    }

    return formattedTools;
  }

  public String getUserPrompt() {
    return userPrompt;
  }


  public String getGroundings() {
    return groundings;
  }

  public String getUserInstructions() {
    return userInstructions;
  }

}
