/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */

package com.mulesoft.modules.agent.conductor.internal.state;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.exception.ObjectNotFoundException;
import org.mule.runtime.api.lock.LockFactory;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.core.api.util.func.CheckedFunction;

import com.mulesoft.modules.agent.conductor.internal.state.model.TaskContext;

import java.util.concurrent.locks.Lock;

import javax.inject.Inject;

import org.slf4j.Logger;

public class TaskContextService {

  private static final Logger LOGGER = getLogger(TaskContextService.class);

  @Inject
  private LockFactory lockFactory;

  public void upsertTaskContext(TaskContext taskContext, String configRef, ObjectStore<TaskContext> taskContextStore)
      throws ObjectStoreException {

    withLock(taskContext.getTaskId(), taskContext.getContextId(), configRef, storeKey -> {
      if (taskContextStore.contains(storeKey)) {
        taskContextStore.remove(storeKey);
      }
      taskContextStore.store(storeKey, taskContext);
      return null;
    });
  }

  public TaskContext getOrCreate(String taskId, String contextId, String configRefName, ObjectStore<TaskContext> taskContextStore)
      throws ObjectNotFoundException, ObjectStoreException {

    return withLock(taskId, contextId, configRefName, storeKey -> {
      if (taskContextStore.contains(storeKey)) {
        return taskContextStore.retrieve(storeKey);
      }
      return new TaskContext(storeKey, taskId, contextId);
    });
  }

  public void remove(String taskId, String contextId, String configRefName, ObjectStore<TaskContext> taskContextStore)
      throws ObjectStoreException {
    LOGGER.debug("Clearing task context for contextId {}", contextId);
    withLock(taskId, contextId, configRefName, storeKey -> {
      try {
        taskContextStore.remove(storeKey);
      } catch (ObjectNotFoundException e) {
        LOGGER.debug("Could not remove context with id {} because it doesn't exist. Resuming...", contextId);
      }
      return null;
    });
  }

  private <T> T withLock(String taskId, String contextId, String configRef, CheckedFunction<String, T> function)
      throws ObjectStoreException {
    Lock lock = lockFactory.createLock(getLockKey(contextId, configRef));
    lock.lock();
    try {
      return function.apply(getStoreKey(taskId, contextId, configRef));
    } catch (Throwable t) {
      // unwrap checked RuntimeException
      Throwable cause = t.getCause();
      if (cause instanceof ObjectStoreException e) {
        throw e;
      } else {
        throw new ObjectStoreException(cause);
      }
    } finally {
      lock.unlock();
    }
  }

  private String getLockKey(String contextId, String configRefName) {
    return "_loop-state-store-" + configRefName + "-" + contextId;
  }

  private String getStoreKey(String taskId, String contextId, String configRef) {
    return "[" + configRef + "]" + "-" + taskId + "-" + contextId;
  }
}
