/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */
package com.mulesoft.modules.agent.broker.internal.extension;

import static java.util.concurrent.TimeUnit.HOURS;

import static org.mule.runtime.api.meta.Category.SELECT;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;
import static org.mule.sdk.api.meta.JavaVersion.JAVA_17;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.ObjectStoreManager;
import org.mule.runtime.api.store.ObjectStoreSettings;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.Extension;
import org.mule.runtime.extension.api.annotation.Operations;
import org.mule.runtime.extension.api.annotation.connectivity.ConnectionProviders;
import org.mule.runtime.extension.api.annotation.dsl.xml.Xml;
import org.mule.runtime.extension.api.annotation.error.ErrorTypes;
import org.mule.runtime.extension.api.annotation.license.RequiresEnterpriseLicense;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.RefName;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.annotation.param.reference.ObjectStoreReference;
import org.mule.sdk.api.annotation.JavaVersionSupport;

import com.mulesoft.modules.agent.broker.internal.error.BrokerErrorTypes;
import com.mulesoft.modules.agent.broker.internal.extension.connection.einstein.EinsteinConnectionProvider;
import com.mulesoft.modules.agent.broker.internal.extension.connection.openai.OpenAIConnectionProvider;
import com.mulesoft.modules.agent.broker.internal.operation.loop.LoopOperation;
import com.mulesoft.modules.agent.broker.internal.state.model.Conversation;
import com.mulesoft.modules.agent.broker.internal.state.model.TaskContext;

import javax.inject.Inject;

/**
 * Module for Agents orchestration
 * <p>
 * This is a BETA version.
 *
 * @since 0.1.0
 */
@Extension(name = "Agents Broker", category = SELECT)
@JavaVersionSupport(JAVA_17)
@Operations(LoopOperation.class)
@ConnectionProviders({EinsteinConnectionProvider.class, OpenAIConnectionProvider.class})
@RequiresEnterpriseLicense(allowEvaluationLicense = true)
@ErrorTypes(BrokerErrorTypes.class)
@Xml(prefix = "agents-broker")
public class AgentsBroker implements Initialisable, Startable, Stoppable {

  @Inject
  private ObjectStoreManager objectStoreManager;

  @RefName
  private String configName;

  @Parameter
  @Optional(defaultValue = "25")
  @Expression(NOT_SUPPORTED)
  @DisplayName("Max Number of Loops")
  @Summary("The maximum number of iterations to control the flow's execution.")
  private Integer maxNumberOfLoops;

  @Parameter
  @Optional(defaultValue = "3")
  @Expression(NOT_SUPPORTED)
  @DisplayName("Max Consecutive Errors")
  @Summary("The maximum number of errors that the orchestrator will attempt to self-heal before returning a failed status.")
  private Integer maxConsecutiveErrors;

  @Parameter
  @Optional
  @Expression(NOT_SUPPORTED)
  @DisplayName("Conversation ObjectStore")
  @Summary("Object Store for storing conversations state.")
  @ObjectStoreReference
  private ObjectStore conversationObjectStore;

  @Parameter
  @Optional
  @Expression(NOT_SUPPORTED)
  @DisplayName("Tasks ObjectStore")
  @Summary("Object Store for storing Tasks states.")
  @ObjectStoreReference
  private ObjectStore tasksObjectStore;

  @Override
  public void initialise() throws InitialisationException {
    if (conversationObjectStore == null) {
      conversationObjectStore = objectStoreManager.getOrCreateObjectStore("_agentBrokerModule_" + configName
          + "_conversations", ObjectStoreSettings.builder()
              .persistent(true)
              .entryTtl(HOURS.toMillis(1))
              .build());
      try {
        conversationObjectStore.open();
      } catch (ObjectStoreException e) {
        throw new InitialisationException(e, this);
      }
    }

    if (tasksObjectStore == null) {
      tasksObjectStore = objectStoreManager.getOrCreateObjectStore("_agentBrokerModule_" + configName
          + "_tasks", ObjectStoreSettings.builder()
              .persistent(true)
              .entryTtl(HOURS.toMillis(1))
              .build());
      try {
        tasksObjectStore.open();
      } catch (ObjectStoreException e) {
        throw new InitialisationException(e, this);
      }
    }
  }

  @Override
  public void start() throws MuleException {
    startIfNeeded(conversationObjectStore);
    startIfNeeded(tasksObjectStore);
  }

  @Override
  public void stop() throws MuleException {
    stopIfNeeded(conversationObjectStore);
    stopIfNeeded(tasksObjectStore);
  }

  public ObjectStore<Conversation> getConversationStore() {
    return conversationObjectStore;
  }

  public ObjectStore<TaskContext> getTasksObjectStore() {
    return tasksObjectStore;
  }

  public String getConfigName() {
    return configName;
  }
}

