/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */
package com.mulesoft.modules.agent.broker.internal.extension.connection.openai;

import static org.mule.runtime.api.connection.ConnectionValidationResult.success;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.mule.runtime.extension.api.annotation.param.display.Placement.SECURITY_TAB;

import org.mule.runtime.api.connection.CachedConnectionProvider;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.connectivity.NoConnectivityTest;

import com.mulesoft.modules.agent.broker.internal.extension.connection.BaseConnectionProvider;
import com.mulesoft.modules.agent.broker.internal.extension.connection.LLMClient;

import com.openai.client.okhttp.OpenAIOkHttpClient;

@Alias("openai")
public class OpenAIConnectionProvider extends BaseConnectionProvider
    implements CachedConnectionProvider<LLMClient>, NoConnectivityTest, Initialisable {

  @ParameterGroup(name = "OpenAI Settings")
  private OpenAISettings openAISettings;

  @Parameter
  @Optional
  @Expression(NOT_SUPPORTED)
  @Placement(tab = SECURITY_TAB)
  private TlsContextFactory tlsContextFactory;

  @Override
  public void initialise() throws InitialisationException {
    initialiseIfNeeded(tlsContextFactory);
  }

  @Override
  public LLMClient connect() throws ConnectionException {
    OpenAIOkHttpClient.Builder builder = OpenAIOkHttpClient.builder()
        .apiKey(openAISettings.getApiKey());

    if (openAISettings.getUrl() != null) {
      builder.baseUrl(openAISettings.getUrl());
    }

    if (tlsContextFactory != null) {
      if (tlsContextFactory.getTrustStoreConfiguration().isInsecure()) {
        try {
          builder.sslSocketFactory(tlsContextFactory.createSocketFactory()).trustManager(new InsecureTrustManager());
        } catch (Exception e) {
          throw new MuleRuntimeException(createStaticMessage("Could not create SSL Socket factory"), e);
        }
      } else {
        LOGGER.info("Secure custom TLS context is not supported for OpenAI. Ignoring and using default instead");
      }
    }

    return new OpenAIClient(builder.build(), openAISettings);
  }

  @Override
  public ConnectionValidationResult validate(LLMClient llmClient) {
    return success();
  }
}
