/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */
package com.mulesoft.modules.agent.broker.internal.serializer;

import static org.mule.runtime.api.metadata.DataType.STRING;
import static org.mule.runtime.api.metadata.DataType.fromObject;
import static org.mule.runtime.api.metadata.DataType.fromType;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JSON;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.mule.runtime.extension.api.runtime.operation.Result;

import com.mulesoft.modules.agent.broker.internal.state.model.LLMOutput;

import java.io.InputStream;

import javax.inject.Inject;

public class SerializationService implements Initialisable {

  private static final String PAYLOAD = "payload";
  private static final DataType LLM_OUTPUT_DATA_TYPE = fromType(LLMOutput.class);
  private static final DataType JSON_STREAM = DataType.builder().type(InputStream.class).mediaType(APPLICATION_JSON).build();

  @Inject
  protected ExpressionManager expressionManager;

  private DwConverter einsteinOutputReader;
  private DwConverter toStringReader;

  public LLMOutput parseEinsteinOutput(InputStream json) {
    try {
      return einsteinOutputReader.evaluate(json, LLM_OUTPUT_DATA_TYPE);
    } catch (Exception e) {
      throw new MuleRuntimeException(e);
    }
  }

  public <T, A> String asString(Result<T, A> result) {
    return toStringReader.evaluateAsString(result.getOutput(), STRING);
  }

  @Override
  public void initialise() {
    einsteinOutputReader = new DwConverter(expressionManager,
                                           """
                                               %dw 2.0
                                               output application/java

                                               var response = read(payload.response, "json")
                                               fun parse(value) = (if ((typeOf(value) as String) == "Object") (
                                                 write(value, "json") as String) else (write(value, "text") as String))
                                               ---
                                               {
                                                   toolToCall: response.toolToCall default null,
                                                   toolInput: parse(response.toolInput),
                                                   result: parse(response.result),
                                                   reasoning: response.reasoning default "",
                                                   goalComplete: response.goalComplete default false,
                                                   additionalInputNeeded: response.additionalInputNeeded default false
                                               }
                                               """,
                                           (value, builder) -> builder.addBinding("payload", new TypedValue(value, JSON_STREAM)));

    toStringReader = new DwConverter(expressionManager,
                                     """
                                         output application/java
                                         ---
                                         payload as String
                                         """,
                                     (value, builder) -> builder.addBinding(PAYLOAD, new TypedValue(value, fromObject(value))));
  }
}
