/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */
package com.mulesoft.modules.agent.broker.internal.extension.connection.openai;

import static com.mulesoft.modules.agent.broker.internal.error.BrokerErrorTypes.REASONING_ERROR;

import static io.a2a.util.Utils.OBJECT_MAPPER;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.extension.api.exception.ModuleException;

import com.mulesoft.modules.agent.broker.internal.extension.connection.LLMClient;
import com.mulesoft.modules.agent.broker.internal.llm.LLMOrchestrationRequest;
import com.mulesoft.modules.agent.broker.internal.state.model.LLMOutput;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.openai.models.Reasoning;
import com.openai.models.ReasoningEffort;
import com.openai.models.responses.ResponseCreateParams;

public class OpenAIClient implements LLMClient {

  private com.openai.client.OpenAIClient client;
  private final OpenAISettings settings;

  public OpenAIClient(com.openai.client.OpenAIClient client, OpenAISettings settings) {
    this.client = client;
    this.settings = settings;
  }

  @Override
  public CompletableFuture<LLMOutput> reasonNextStep(LLMOrchestrationRequest request) {
    ResponseCreateParams createParams = ResponseCreateParams.builder()
        .input(request.getPrompt())
        .model(settings.getModelName())
        .reasoning(Reasoning.builder().effort(ReasoningEffort.of(settings.getReasoningEffort().name().toLowerCase())).build())
        .additionalHeaders(Map.of("X-ANYPOINT-MODEL", List.of(settings.getModelName())))
        .build();

    return client.async().responses().create(createParams)
        .thenApply(response -> response.output().stream()
            .flatMap(item -> item.message().stream())
            .flatMap(message -> message.content().stream())
            .flatMap(content -> content.outputText().stream())
            .findFirst()
            .map(output -> {
              try {
                return OBJECT_MAPPER.readValue(output.text(), LLMOutput.class);
              } catch (Exception e) {
                throw new ModuleException("LLM provided unparseable response", REASONING_ERROR, e);
              }
            })
            .orElseThrow(() -> new MuleRuntimeException(createStaticMessage("LLM provided wrong response format"))));
  }

  @Override
  public void close() {
    client.close();
  }
}
