/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */
package com.mulesoft.modules.agent.broker.internal.extension.connection.openai;

import static com.mulesoft.modules.agent.broker.internal.error.BrokerErrorTypes.TOOL_ERROR;
import static com.mulesoft.modules.agent.broker.internal.util.ToolUtils.DEFAULT_TOOL_METADATA_EXPIRY;
import static com.mulesoft.modules.agent.broker.internal.util.ToolUtils.newToolCache;
import static com.mulesoft.modules.agent.broker.internal.util.ToolUtils.toOpenAiToolSchema;

import static com.openai.models.responses.Tool.ofCustom;
import static com.openai.models.responses.Tool.ofFunction;

import org.mule.runtime.api.util.Reference;
import org.mule.runtime.extension.api.exception.ModuleException;

import com.mulesoft.modules.agent.broker.internal.extension.connection.LLMClient;
import com.mulesoft.modules.agent.broker.internal.extension.connection.session.FunctionBasedSession;
import com.mulesoft.modules.agent.broker.internal.extension.connection.session.StructuredResponseSession;
import com.mulesoft.modules.agent.broker.internal.llm.LLMRequest;
import com.mulesoft.modules.agent.broker.internal.operation.loop.LoopOperation;
import com.mulesoft.modules.agent.broker.internal.tool.Tool;
import com.mulesoft.modules.agent.broker.internal.tool.ToolVisitor;
import com.mulesoft.modules.agent.broker.internal.tool.a2a.A2AService.A2ATool;
import com.mulesoft.modules.agent.broker.internal.tool.mcp.McpService.McpTool;

import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.openai.core.JsonValue;
import com.openai.models.responses.CustomTool;
import com.openai.models.responses.FunctionTool;

public class OpenAIClient implements LLMClient {

  private com.openai.client.OpenAIClient client;
  private final OpenAISettings settings;

  private final LoadingCache<Reference<Tool>, com.openai.models.responses.Tool> toolAdapters;

  public OpenAIClient(com.openai.client.OpenAIClient client, OpenAISettings settings) {
    this.client = client;
    this.settings = settings;
    // the default refresh rate + 1 minute for queued up requests
    toolAdapters = newToolCache(ref -> adapt(ref.get()), DEFAULT_TOOL_METADATA_EXPIRY.plusMinutes(1));
  }

  @Override
  public LLMSession createSession(LLMRequest request) {
    var tools = adapt(request.getTools());
    var model = settings.getModelName();

    if (model.contains("flash") || model.contains("gemini")) {
      return new FunctionBasedSession(client, request, settings, tools);
    }

    return new StructuredResponseSession(client, request, settings, tools);
  }

  @Override
  public void close() {
    client.close();
  }

  private List<com.openai.models.responses.Tool> adapt(List<Tool> tools) {
    return tools.stream()
        .map(tool -> toolAdapters.get(new Reference<>(tool)))
        .toList();
  }

  private com.openai.models.responses.Tool adapt(Tool tool) {
    Reference<com.openai.models.responses.Tool> retVal = new Reference<>();

    tool.accept(new ToolVisitor() {

      @Override
      public void visit(McpTool mcpTool) {
        retVal.set(ofFunction(FunctionTool.builder()
            .name(mcpTool.getId())
            .description(mcpTool.getDescription())
            .strict(true)
            .parameters(FunctionTool.Parameters.builder()
                .putAllAdditionalProperties(parseInputSchema(mcpTool))
                .build())
            .build()));
      }

      @Override
      public void visit(A2ATool tool) {
        retVal.set(ofCustom(CustomTool.builder()
            .name(tool.getId())
            .description("This tool is an external agent. Use a concise natural language prompt clearly stating what is needed from this tool. Only include information relevant for this taskHere's the agent's description: "
                + tool.getDescription())
            .build()));
      }

      @Override
      public void visit(LoopOperation.CustomTool tool) {
        throw new UnsupportedOperationException();
      }
    });

    return retVal.get();
  }

  private Map<String, JsonValue> parseInputSchema(McpTool tool) {
    try {
      return toOpenAiToolSchema(tool.getInput());
    } catch (JsonProcessingException e) {
      throw new ModuleException("Supplied schema for MCP tool " + tool.getName() + " is invalid", TOOL_ERROR, e);
    }
  }

}
