/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */
package com.mulesoft.modules.agent.broker.internal.extension.connection.session;

import static com.mulesoft.modules.agent.broker.internal.error.BrokerErrorTypes.LLM_ERROR;

import static com.openai.models.responses.ResponseIncludable.REASONING_ENCRYPTED_CONTENT;
import static com.openai.models.responses.ResponseInputItem.Message.Role.USER;
import static com.openai.models.responses.ResponseInputItem.ofCustomToolCall;
import static com.openai.models.responses.ResponseInputItem.ofCustomToolCallOutput;
import static com.openai.models.responses.ResponseInputItem.ofFunctionCall;
import static com.openai.models.responses.ResponseInputItem.ofFunctionCallOutput;
import static com.openai.models.responses.ResponseInputItem.ofMessage;
import static com.openai.models.responses.ResponseInputItem.ofReasoning;
import static com.openai.models.responses.ResponseInputItem.ofResponseOutputMessage;
import static io.a2a.util.Utils.OBJECT_MAPPER;
import static org.mule.runtime.api.functional.Either.right;

import org.mule.runtime.api.functional.Either;
import org.mule.runtime.extension.api.exception.ModuleException;

import com.mulesoft.modules.agent.broker.internal.extension.connection.LLMClient.LLMSession;
import com.mulesoft.modules.agent.broker.internal.extension.connection.openai.OpenAISettings;
import com.mulesoft.modules.agent.broker.internal.llm.LLMRequest;
import com.mulesoft.modules.agent.broker.internal.state.model.HasInternalReasoning;
import com.mulesoft.modules.agent.broker.internal.state.model.InternalReasoning;
import com.mulesoft.modules.agent.broker.internal.state.model.Iteration;
import com.mulesoft.modules.agent.broker.internal.state.model.LLMOutput;
import com.mulesoft.modules.agent.broker.internal.state.model.ToolSelection;
import com.mulesoft.modules.agent.broker.internal.tool.ToolResponse;
import com.mulesoft.modules.agent.broker.internal.tool.ToolType;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.openai.client.OpenAIClient;
import com.openai.core.JsonString;
import com.openai.models.Reasoning;
import com.openai.models.ReasoningEffort;
import com.openai.models.responses.ResponseCreateParams;
import com.openai.models.responses.ResponseCustomToolCall;
import com.openai.models.responses.ResponseCustomToolCallOutput;
import com.openai.models.responses.ResponseFunctionToolCall;
import com.openai.models.responses.ResponseInputItem;
import com.openai.models.responses.ResponseOutputMessage;
import com.openai.models.responses.ResponseOutputText;
import com.openai.models.responses.ResponseReasoningItem;
import com.openai.models.responses.Tool;

public abstract class BaseLLMSession implements LLMSession {

  private final LLMRequest llmRequest;
  private final OpenAISettings settings;
  private final List<Tool> tools;

  protected final ResponseCreateParams.Builder builder;
  protected final List<ResponseInputItem> inputs = new LinkedList<>();
  protected final OpenAIClient client;
  protected InternalReasoning internalReasoning;

  public BaseLLMSession(OpenAIClient client, LLMRequest llmRequest, OpenAISettings settings, List<Tool> tools) {
    this.client = client;
    this.llmRequest = llmRequest;
    this.settings = settings;
    this.tools = tools;
    builder = newRequestBuilder();
  }

  @Override
  public void addIteration(Iteration iteration) {
    if (iteration.getUserPrompt() != null) {
      inputs.add(ofMessage(ResponseInputItem.Message.builder()
          .role(USER)
          .addInputTextContent(iteration.getUserPrompt())
          .build()));
    }

    var toolSelection = iteration.getToolSelection();
    if (toolSelection != null) {
      addInternalReasoning(toolSelection);
      inputs.add(ofFunctionCall(ResponseFunctionToolCall.builder()
          .callId(toolSelection.getSelectionId())
          .arguments(toolSelection.getInput())
          .name(toolSelection.getToolId())
          .build()));
    }

    addToolResponse(iteration.getToolResponse());

    var llmOutput = iteration.getLlmOutput();
    if (llmOutput != null) {
      addInternalReasoning(llmOutput);
      try {
        inputs.add(ofResponseOutputMessage(ResponseOutputMessage.builder()
            .id(llmOutput.getId())
            .role(JsonString.of("assistant"))
            .status(ResponseOutputMessage.Status.COMPLETED)
            .addContent(ResponseOutputText.builder()
                .text(OBJECT_MAPPER.writeValueAsString(llmOutput))
                .annotations(List.of())
                .build())
            .build()));
      } catch (JsonProcessingException e) {
        throw new ModuleException("Exception serializing response output", LLM_ERROR, e);
      }
    }
  }

  @Override
  public void addToolResponse(ToolResponse toolResponse) {
    if (toolResponse != null) {
      if (toolResponse.getToolType() == ToolType.MCP) {
        inputs.add(ofFunctionCallOutput(ResponseInputItem.FunctionCallOutput.builder()
            .outputAsJson(toolResponse.getResult())
            .callId(toolResponse.getSelection().getSelectionId())
            .status(ResponseInputItem.FunctionCallOutput.Status.COMPLETED)
            .build()));
      } else if (toolResponse.getToolType() == ToolType.A2A) {
        inputs.add(ofCustomToolCallOutput(ResponseCustomToolCallOutput.builder()
            .output(toolResponse.getResult())
            .callId(toolResponse.getSelection().getSelectionId())
            .build()));
      }
    }
  }

  private void addInternalReasoning(HasInternalReasoning reasoned) {
    var internalReasoning = reasoned != null ? reasoned.getInternalReasoning() : null;
    if (internalReasoning != null) {
      inputs.add(ofReasoning(ResponseReasoningItem.builder()
          .id(internalReasoning.getId())
          .encryptedContent(internalReasoning.getReasoning())
          .summary(List.of())
          .build()));
    }
  }

  protected abstract CompletableFuture<Either<LLMOutput, ToolSelection>> doGetNext();

  @Override
  public final CompletableFuture<Either<LLMOutput, ToolSelection>> getNext() {
    builder.inputOfResponse(inputs);
    return doGetNext();
  }

  protected Either<LLMOutput, ToolSelection> onCustomCall(ResponseCustomToolCall toolCall) {
    inputs.add(ofCustomToolCall(toolCall));

    return right(new ToolSelection(
                                   toolCall.callId(),
                                   toolCall.name(),
                                   toolCall.input(),
                                   internalReasoning));
  }

  protected Either<LLMOutput, ToolSelection> onFunctionCall(ResponseFunctionToolCall toolCall) {
    inputs.add(ofFunctionCall(toolCall));

    return right(new ToolSelection(
                                   toolCall.callId(),
                                   toolCall.name(),
                                   toolCall.arguments(),
                                   internalReasoning));
  }

  protected void onReasoningItem(ResponseReasoningItem reasoningItem) {
    inputs.add(ofReasoning(reasoningItem));
    internalReasoning = new InternalReasoning(reasoningItem.id(), reasoningItem.encryptedContent().orElse(""));
  }

  @Override
  public LLMRequest getInitialRequest() {
    return llmRequest;
  }

  protected ResponseCreateParams.Builder newRequestBuilder() {
    return ResponseCreateParams.builder()
        .instructions(buildSystemPrompt(llmRequest))
        .model(settings.getModelName())
        .parallelToolCalls(false)
        .store(false)
        .reasoning(Reasoning.builder().effort(ReasoningEffort.of(settings.getReasoningEffort().name().toLowerCase())).build())
        .include(List.of(REASONING_ENCRYPTED_CONTENT))
        .tools(tools)
        .additionalHeaders(Map.of("X-ANYPOINT-MODEL", List.of(settings.getModelName())));
  }

  private String buildSystemPrompt(LLMRequest request) {
    StringBuilder prompt =
        new StringBuilder("""
            You are a task decomposition expert that analyzes user requests, identifies required sub-tasks, selects appropriate tools, and synthesizes final answers.

            1. **Decompose** the query into atomic sub-tasks
            2. **Match** each sub-task to the appropriate tool below
            3. **Execute** tools in optimal sequence
            4. **Synthesize** results into final response

            Here is an Example of how to break down a user prompt
            **User Query:** 'Analyze Q2 earnings for Tesla and compare to Ford in EUR'
            **Sub-tasks:**
            1. Get Tesla financials (USD) → Financial Summary Tool
            2. Get Ford financials (USD) → Financial Summary Tool
            3. Convert USD figures to EUR → Currency Converter Tool
            4. Perform comparative analysis → Built-in Analysis Module

            The User's instructions section contains directives *YOU MUST* follow when deciding which action to take next.

            ### User's instructions

            """)
            .append(request.getInstructions())
            .append("""

                ### Instructions for executing steps, selecting tools and generating output

                - Execute the list of steps in order. For each step, determine if invoking a tool is necessary
                - When you reach a step that requires a tool, look at the available tools and conversation history to determine the *single best tool* to call next.

                ### Constraints

                - Use the conversation history to avoid redundant tool calls and to track progress toward the goal.
                """);

    return prompt.toString();
  }
}
