/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */
package com.mulesoft.modules.agent.broker.internal.extension.connection.session;

import static com.mulesoft.modules.agent.broker.internal.error.BrokerErrorTypes.REASONING_ERROR;

import static com.openai.models.responses.ResponseInputItem.ofResponseOutputMessage;
import static org.mule.runtime.api.functional.Either.left;

import org.mule.runtime.api.functional.Either;
import org.mule.runtime.extension.api.exception.ModuleException;

import com.mulesoft.modules.agent.broker.internal.extension.connection.openai.OpenAISettings;
import com.mulesoft.modules.agent.broker.internal.llm.LLMRequest;
import com.mulesoft.modules.agent.broker.internal.state.model.LLMOutput;
import com.mulesoft.modules.agent.broker.internal.state.model.ToolSelection;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.openai.client.OpenAIClient;
import com.openai.models.responses.ResponseCreateParams;
import com.openai.models.responses.StructuredResponseCreateParams;
import com.openai.models.responses.StructuredResponseOutputItem;
import com.openai.models.responses.Tool;

public class StructuredResponseSession extends BaseLLMSession {

  private StructuredResponseCreateParams.Builder<LLMOutput> structuredBuilder;

  public StructuredResponseSession(OpenAIClient client, LLMRequest llmRequest, OpenAISettings settings, List<Tool> tools) {
    super(client, llmRequest, settings, tools);
  }

  @Override
  protected ResponseCreateParams.Builder newRequestBuilder() {
    var builder = super.newRequestBuilder();
    structuredBuilder = builder.text(LLMOutput.class);

    return builder;
  }

  @Override
  protected CompletableFuture<Either<LLMOutput, ToolSelection>> doGetNext() {
    return client.async().responses().create(structuredBuilder.build()).thenApply(response -> {

      for (StructuredResponseOutputItem<LLMOutput> item : response.output()) {
        if (item.isReasoning()) {
          onReasoningItem(item.asReasoning());
        } else if (item.isFunctionCall()) {
          return onFunctionCall(item.asFunctionCall());
        } else if (item.isCustomToolCall()) {
          return onCustomCall(item.asCustomToolCall());
        } else if (item.isMessage()) {
          var message = item.asMessage();
          inputs.add(ofResponseOutputMessage(message.rawMessage()));

          LLMOutput output = message.content().stream()
              .flatMap(content -> content.outputText().stream())
              .findFirst()
              .orElseThrow(() -> new ModuleException("LLM provided wrong response format", REASONING_ERROR));

          output.setInternalReasoning(internalReasoning);
          output.setId(message.rawMessage().id());
          return left(output);
        }
      }

      throw new ModuleException("LLM didn't provide a valid response", REASONING_ERROR);
    });
  }
}
