/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */
package com.mulesoft.modules.agent.broker.internal.util;

import static java.time.Duration.ofHours;
import static java.util.regex.Pattern.compile;

import static com.openai.core.JsonValue.fromJsonNode;
import static io.a2a.util.Utils.OBJECT_MAPPER;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.core.api.util.StringUtils.isBlank;

import org.mule.runtime.api.exception.MuleRuntimeException;

import com.mulesoft.modules.agent.broker.internal.state.model.LLMOutput;

import java.time.Duration;
import java.util.Map;
import java.util.function.Function;
import java.util.regex.Pattern;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.BooleanNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.module.jsonSchema.JsonSchemaGenerator;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.openai.core.JsonValue;
import org.apache.commons.text.RandomStringGenerator;

public final class ToolUtils {

  public static final Duration DEFAULT_TOOL_METADATA_EXPIRY = ofHours(1);

  // leave some room for flex to add its own prefixes
  static final int MAX_TOOL_ID_LENGTH = 50;
  private static final Pattern TOOL_NAME_PATTERN = compile("^[a-zA-Z0-9_-]+$");
  private static final Pattern TOOL_NAME_CLEANER = compile("[^a-zA-Z0-9_-]");
  private static final String BASE62_CHARS = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
  private static final RandomStringGenerator TOOL_CALL_ID_GENERATOR = RandomStringGenerator.builder()
      .selectFrom("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz".toCharArray())
      .get();

  public static String generateToolId(final String configName, String toolName) {
    if (isBlank(toolName)) {
      throw new IllegalArgumentException("tool name cannot be blank");
    }

    if (!TOOL_NAME_PATTERN.matcher(toolName).matches()) {
      toolName = TOOL_NAME_CLEANER.matcher(toolName).replaceAll("_");
    }

    final var hash = generateHash(configName + "." + toolName);
    String id = hash + "_" + toolName;

    if (id.length() > MAX_TOOL_ID_LENGTH) {
      id = id.substring(0, MAX_TOOL_ID_LENGTH);
    }

    return id;
  }

  public static <K, V> Cache<K, V> newToolCache() {
    return newToolCache(DEFAULT_TOOL_METADATA_EXPIRY);
  }

  public static <K, V> Cache<K, V> newToolCache(Duration expiry) {
    return Caffeine.newBuilder()
        .expireAfterWrite(expiry)
        .build();
  }

  public static String randomToolCallId() {
    return "call_" + randomString(24);
  }

  public static String randomString(int length) {
    return TOOL_CALL_ID_GENERATOR.generate(length);
  }

  public static <K, V> LoadingCache<K, V> newToolCache(Function<K, V> loader, Duration expiry) {
    return Caffeine.newBuilder()
        .expireAfterWrite(expiry)
        .build(loader::apply);
  }

  /**
   * Using this instead of {@link String#hashCode()} to avoid issue when two replicas are running different JVM versions
   */
  private static String generateHash(String input) {
    // Use a combination of FNV-1a and polynomial rolling hash
    long hash1 = 0xcbf29ce484222325L; // FNV offset basis
    long hash2 = 0L;
    long prime = 31L;
    long mod = 1000000000000L; // Keep numbers manageable

    for (int i = 0; i < input.length(); i++) {
      char c = input.charAt(i);

      // FNV-1a hash
      hash1 ^= c;
      hash1 *= 0x100000001b3L; // FNV prime

      // Polynomial rolling hash
      hash2 = (hash2 * prime + c) % mod;
    }

    // Combine both hashes with additional mixing
    long combined = hash1 ^ (hash2 << 32) ^ (hash2 >>> 32);

    // Additional mixing to improve distribution
    combined ^= (combined >>> 33);
    combined *= 0xff51afd7ed558ccdL;
    combined ^= (combined >>> 33);
    combined *= 0xc4ceb9fe1a85ec53L;
    combined ^= (combined >>> 33);

    // Convert to base-62 (alphanumeric) for compact representation
    return toBase62(Math.abs(combined), 10);
  }

  private static String toBase62(long num, int length) {
    StringBuilder sb = new StringBuilder();

    while (sb.length() < length) {
      sb.append(BASE62_CHARS.charAt((int) (num % 62)));
      num /= 62;
    }

    return sb.reverse().toString();
  }

  public static Map<String, JsonValue> openAiLLMOutputSchema() {
    try {
      ObjectNode schema = OBJECT_MAPPER.valueToTree(new JsonSchemaGenerator(OBJECT_MAPPER).generateSchema(LLMOutput.class));

      schema.put("$schema", "http://json-schema.org/draft-07/schema#");
      schema.put("description", "Schema for a n LLM structured answer");

      return toOpenAiToolSchema(schema);
    } catch (JsonProcessingException e) {
      throw new MuleRuntimeException(createStaticMessage("Unable to generate schema for a n LLM structured answer"), e);
    }
  }

  public static Map<String, JsonValue> toOpenAiToolSchema(String schema) throws JsonProcessingException {
    return toOpenAiToolSchema((ObjectNode) OBJECT_MAPPER.readTree(schema));
  }

  public static Map<String, JsonValue> toOpenAiToolSchema(ObjectNode schemaRoot) throws JsonProcessingException {
    schemaRoot.set("additionalProperties", BooleanNode.getFalse());
    schemaRoot.set("required", collectRequiredProperties(schemaRoot));

    return (Map<String, JsonValue>) fromJsonNode(schemaRoot).asObject().get();
  }

  private static ArrayNode collectRequiredProperties(ObjectNode root) {
    final ArrayNode required = OBJECT_MAPPER.createArrayNode();
    JsonNode node = root.get("properties");
    if (node == null) {
      return required;
    }

    if (!node.isObject()) {
      throw new IllegalArgumentException("'properties' is not an object");
    }

    node.fieldNames().forEachRemaining(required::add);

    return required;
  }


  private ToolUtils() {}
}
