/*
 * (c) 2025 MuleSoft, Inc. The software in this package is published under the terms of the Commercial Free Software license V.1 a copy of which has been included with this distribution in the LICENSE.md file.
 */
package com.mulesoft.modules.agent.broker.internal.util;

import static com.mulesoft.modules.agent.broker.internal.util.ToolUtils.MAX_TOOL_ID_LENGTH;
import static com.mulesoft.modules.agent.broker.internal.util.ToolUtils.generateToolId;
import static com.mulesoft.modules.agent.broker.internal.util.ToolUtils.randomString;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.lessThanOrEqualTo;

import java.util.HashSet;

import org.apache.commons.text.RandomStringGenerator;
import org.junit.Before;
import org.junit.Test;

public class ToolNameUtilsTest {

  @Test
  public void toolIdOnlyHasValidChars() {
    assertValidCharsOnly("my-tool", "my-tool");
    assertValidCharsOnly("my tool", "my_tool");
    assertValidCharsOnly("my.tool  mola", "my_tool__mola");
  }

  @Test
  public void toolIdHasCorrectSize() {
    final var configName = randomString(40);

    assertIdSize(configName, randomString(40));

    // now try with something within limits
    assertIdSize(configName, randomString(5));
  }

  @Test
  public void toolIdIdempotency() {
    final var configName = randomString(40);
    final var toolName = randomString(40);

    var id1 = generateToolId(configName, toolName);
    var id2 = generateToolId(configName, toolName);

    assertThat(id1, equalTo(id2));
  }

  @Test
  public void toolIdCollisions() {
    final int testCount = 100;
    final var hashes = new HashSet<String>();

    for (int i = 0; i < testCount; i++) {
      hashes.add(generateToolId(randomString(40), randomString(40)));
    }

    assertThat(hashes, hasSize(testCount));
  }

  private static void assertIdSize(String configName, String toolName) {
    var id = generateToolId(configName, toolName);
    assertThat(id.length(), lessThanOrEqualTo(MAX_TOOL_ID_LENGTH));
    assertThat(toolName.contains((id.substring(11))), is(true));
  }

  private void assertValidCharsOnly(String toolId, String expected) {
    String actual = generateToolId("", toolId).substring(10);
    assertThat(actual, equalTo("_" + expected));
  }
}
