/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.modules.interceptor.spring;

import java.util.Map;

import org.springframework.beans.factory.config.ConstructorArgumentValues;
import org.springframework.beans.factory.support.AbstractBeanDefinition;


/**
 * <p>
 * Sets the constructor information to a bean definition
 * </p>
 *
 * @author Mulesoft Inc.
 * @since  3.3.2
 */
public class BeanFactoryMethodBuilder
{

    /**
     * <p>
     * The bean definition that has to be modified
     * </p>
     */
    private AbstractBeanDefinition beanDefinition;


    /**
     * <p>
     * Constructor. Sets the factory method name and the bean factory.
     * </p>
     *
     * @param beanDefinition    <p>
     *                          The bean definition that has to be modified
     *                          </p>
     * @param factoryMethodName <p>
     *                          The factory method name
     *                          </p>
     * @param beanFactoryName   <p>
     *                          The bean factory name
     *                          </p>
     */
    public BeanFactoryMethodBuilder(AbstractBeanDefinition beanDefinition,
                                    String factoryMethodName,
                                    String beanFactoryName)
    {
        this.beanDefinition = beanDefinition;

        beanDefinition.setFactoryBeanName(beanFactoryName);
        beanDefinition.setFactoryMethodName(factoryMethodName);
    }

    /**
     * <p>
     * Sets the constructor arguments to the bean definition
     * </p>
     *
     * @param constructorArguments <p>
     *                             The constructor arguments for the bean definition
     *                             </p>
     * @return <p>
     *         The bean definition with the one it was created.
     *         </p>
     */
    public AbstractBeanDefinition withConstructorArguments(Object... constructorArguments)
    {
        int argumentsSize = constructorArguments.length;
        ConstructorArgumentValues constructorArgumentValues = beanDefinition.getConstructorArgumentValues();

        ConstructorArgumentValues values = new ConstructorArgumentValues();
        Map<Integer, ConstructorArgumentValues.ValueHolder> indexedArgumentValues = constructorArgumentValues.getIndexedArgumentValues();


        for (int i = 0; i < argumentsSize; i++)
        {
            values.addIndexedArgumentValue(i, constructorArguments[i]);
        }

        for (int i = 0; i < indexedArgumentValues.size(); i++)
        {
            values.addIndexedArgumentValue(i + argumentsSize, indexedArgumentValues.get(i));
        }

        beanDefinition.setConstructorArgumentValues(values);

        return beanDefinition;
    }
}

