/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package org.mule.module.dynamicscrm.paging;

import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.api.MuleException;
import org.mule.common.query.DsqlQuery;
import org.mule.module.dynamicscrm.paging.exception.DynamicsCrmPagingException;
import org.mule.module.dynamicscrm.query.DynamicsCrmQueryVisitor;
import org.mule.module.dynamicscrm.utils.DynamicsCrmUtils;
import org.mule.streaming.PagingDelegate;

import com.microsoft.schemas.xrm._2011.contracts.EntityCollection;
import com.microsoft.schemas.xrm._2011.contracts.FetchExpression;
import com.microsoft.schemas.xrm._2011.contracts.services.IOrganizationService;
import com.microsoft.schemas.xrm._2011.contracts.services.IOrganizationServiceRetrieveMultipleOrganizationServiceFaultFaultFaultMessage;

public class RetrieveMultipleByQueryPagingDelegate extends PagingDelegate<Map<String, Object>> {

	static final private Log logger = LogFactory.getLog(RetrieveMultipleByQueryPagingDelegate.class);
	
	private IOrganizationService client;
	private EntityCollection lastEntityCollection;
	private DsqlQuery lastQuery;
	private boolean firstPage;
	private Integer pageSize;
	private Integer limit;
	private Integer currentLimit;

	public RetrieveMultipleByQueryPagingDelegate(IOrganizationService client, EntityCollection entityCollection, DsqlQuery query, Integer pageSize) {
		super();
		this.client = client;
		lastEntityCollection = entityCollection;
		lastQuery = query;
		this.pageSize = pageSize;
		firstPage = true;
		
		limit = (query.getLimit() > 0) ? query.getLimit() : null;
		currentLimit = 0;
		
	}

	@Override
	public void close() throws MuleException {
		if (lastEntityCollection != null) {
			lastEntityCollection = null;
		}

		if (lastQuery != null) {
			lastQuery = null;
		}
		
		client = null;
	}

	@Override
	public List<Map<String, Object>> getPage() {
		
		logger.debug("Requesting new page");
		
//		// If its the first time getPage is called, we already have the request to the service
		if (firstPage) {
			firstPage = false;
			// The Delegate is initialized with the result of the first page, so no need in calling the service here
			if (lastEntityCollection != null) {
				List<Map<String,Object>> mapEntityCollectionToMapCollection = DynamicsCrmUtils.mapEntityCollectionToMapCollection(lastEntityCollection);
				mapEntityCollectionToMapCollection = trimMapForLimit(mapEntityCollectionToMapCollection);
				addAmmountToCurrentLimit(mapEntityCollectionToMapCollection);
				return mapEntityCollectionToMapCollection;
			} else {
				return null;
			}
		} else if (limit != null && limit > 0 && currentLimit >= limit) {
			return null;
		} else if (lastEntityCollection != null && lastEntityCollection.isMoreRecords() && lastQuery != null) {
			// Modify the query to increase the page number by one
			// If the page is 1 or greater we need to jump to the next page. If not, we were in the first page, and we have to jump to the second
			lastQuery.setOffset(lastQuery.getOffset() > 0 ? lastQuery.getOffset() + 1 : 2);
			
			logger.debug("Calling service for page " + lastQuery.getOffset());
			
			DynamicsCrmQueryVisitor queryVisitor = new DynamicsCrmQueryVisitor(pageSize);
			lastQuery.accept(queryVisitor);
    		String nativeQuery = queryVisitor.getQuery();
			
			FetchExpression query = new FetchExpression();
			query.setQuery(nativeQuery);
			try {
				lastEntityCollection = client.retrieveMultiple(query);
			} catch (IOrganizationServiceRetrieveMultipleOrganizationServiceFaultFaultFaultMessage e) {
				throw new DynamicsCrmPagingException(e);
			}
			
			List<Map<String,Object>> mapEntityCollectionToMapCollection = DynamicsCrmUtils.mapEntityCollectionToMapCollection(lastEntityCollection);
			mapEntityCollectionToMapCollection = trimMapForLimit(mapEntityCollectionToMapCollection);
			addAmmountToCurrentLimit(mapEntityCollectionToMapCollection);
			return mapEntityCollectionToMapCollection;
		}
		
		return null;
	}
	
	private List<Map<String,Object>> trimMapForLimit(List<Map<String,Object>> map) {
		if (limit != null && limit > 0 && map != null) {
			int size = map.size();
			if (currentLimit + size > limit) {
				int trimToSize = limit - currentLimit;
				map = map.subList(0, trimToSize);
			}
		}
		
		return map;
	}
	
	private void addAmmountToCurrentLimit(List<?> list) {
		if (list != null) {
			currentLimit += list.size();
		}
	}

	@Override
	public int getTotalResults() {
		return -1;
	}
}
