/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package org.mule.module.dynamicscrm.query;

import java.util.List;

import org.mule.common.query.DefaultQueryVisitor;
import org.mule.common.query.Field;
import org.mule.common.query.Type;
import org.mule.common.query.expression.Direction;
import org.mule.common.query.expression.OperatorVisitor;
import org.mule.common.query.expression.Value;
import org.mule.module.dynamicscrm.query.exception.DynamicsCrmQueryException;
import org.mule.module.dynamicscrm.query.where.DynamicsCrmWhereManager;

/**
 * Visitor that handles the mapping from a DSQL query into CRM Native Query Language
 */
public class DynamicsCrmQueryVisitor extends DefaultQueryVisitor {

	private Type entity;	
	private List<Field> fields;
	private DynamicsCrmWhereManager where;
	private List<Field> orderBy;
	private Direction orderByDirection;
	private Integer offset;
	private Integer pageSize;
	
	public DynamicsCrmQueryVisitor() {
		where = new DynamicsCrmWhereManager();
		entity = null;
		fields = null;
		orderBy = null;
		offset = null;
		pageSize = null;
	}
	
	public DynamicsCrmQueryVisitor(Integer pageSize) {
		this();
		this.pageSize = pageSize;
	}
	
	public String getQuery() {
		// Minor validations		
		if (entity == null) {
			throw new DynamicsCrmQueryException("The entity for the query must be specified");
		}
		
		if (fields == null) {
			throw new DynamicsCrmQueryException("The fields of the entity for the query must be specified");
		}
		
		StringBuffer queryString = new StringBuffer();
		
		queryString
			.append("<fetch mapping=\"logical\"")
			.append((pageSize != null ? String.format(" count=\"%s\"", pageSize) : ""))
			.append((offset != null ? String.format(" page=\"%s\"", offset) : ""))
			.append(">")
			.append(String.format("<entity name=\"%s\">", entity.getName()));
		
		// Fields
		for (Field field : fields) {
			queryString.append(String.format("<attribute name=\"%s\" />", field.getName()));
		}
		
		// Where
		queryString.append(where.getQueryXml());
		where.release();
		
		// Order By
		if (orderBy != null) {
			for (Field field : orderBy) {
				queryString.append(String.format("<order attribute=\"%s\" descending=\"%s\" />", field.getName(), (orderByDirection.equals(Direction.DESC) ? "true" : "false")));
			}
		}
		
		queryString
			.append("</entity>")
			.append("</fetch>");
		
		return queryString.toString();
	}
		
	@Override
	public void visitTypes(List<Type> types) {
		entity = types.get(0);		
	}
	
	@Override
	public void visitFields(List<Field> fields) {
		this.fields = fields;
	}
	
	@Override
	public OperatorVisitor operatorVisitor() {
		return new DynamicsCrmOperatorVisitor();
	}

	@Override
	public void visitAnd() {
		where.and();
	}

	@Override
	public void visitBeginExpression() {
		where.beginExpression();
	}

	@SuppressWarnings("rawtypes")
	@Override
	public void visitComparison(String operator, Field field, Value value) {
		where.comparison(operator, field.getName(), value.getValue().toString());
	}

	@Override
	public void visitEndPrecedence() {
		where.endPrecedence();
	}

	

	@Override
	public void visitInitPrecedence() {
		where.initPrecedence();
	}

	@Override
	public void visitLimit(int limit) {
		// Limit is unused...
	}

	@Override
	public void visitOR() {
		where.or();
	}

	@Override
	public void visitOffset(int offset) {
		this.offset = new Integer(offset);
	}

	@Override
	public void visitOrderByFields(List<Field> orderByFields, Direction direction) {
		StringBuilder s = new StringBuilder();
		for (Field f : orderByFields) {
			if (s.length() > 0) {
				s.append(", ");
			}
			s.append(fieldToString(f));
		}
		
		orderBy = orderByFields;
		orderByDirection = direction;
	}

	private String fieldToString(Field f) {
		return String.format("{\"name\":\"%s\", \"type\":\"%s\"}", f.getName(), f.getType());
	}
}
