/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package org.mule.module.dynamicscrm.query.where;

import org.mule.module.dynamicscrm.query.exception.DynamicsCrmQueryException;
import org.mule.module.dynamicscrm.query.where.DynamicsCrmWhereComposite.Comparation;

/**
 * Manager for the composite that generates the part of the WHERE for the query.
 * It uses a firstComposite as the head of the WHERE, and a lastComposite as the current position in the composites for aggregation and
 * decendence of levels on endPrecedence 
 */
public class DynamicsCrmWhereManager implements DynamicsCrmWhereComponent {
	
	private DynamicsCrmWhereComponent firstComposite;
	private DynamicsCrmWhereComposite lastComposite;
	
	public DynamicsCrmWhereManager() {
		firstComposite = null;
		lastComposite = null;
	}
	
	/**
	 * A beginExpression means the first \<filter\>. This is managed by the {@link DynamicsCrmWhereComposite} of the composite
	 */
	public void beginExpression() {
		DynamicsCrmWhereComposite dynamicsCrmWhereComposite = new DynamicsCrmWhereComposite(null);
		firstComposite = dynamicsCrmWhereComposite;
		lastComposite = dynamicsCrmWhereComposite;
	}
	
	/**
	 * A new precedence means a new \<Filter\>. This is managed by the {@link DynamicsCrmWhereComposite} of the composite
	 */
	public void initPrecedence() {
		DynamicsCrmWhereComposite dynamicsCrmWhereComposite = new DynamicsCrmWhereComposite(lastComposite);
		lastComposite.addComponent(dynamicsCrmWhereComposite);
		lastComposite = dynamicsCrmWhereComposite;
	}
	
	/**
	 * An end of precedence means go back to the previous \<filter\>
	 */
	public void endPrecedence() {
		lastComposite = lastComposite.getParentComposite();
	}
	
	/**
	 * Define the comparation of the \<filter\> 
	 * @throws DynamicsCrmQueryException By rule, the Dynamics CRM Query Language only allows one comparation per \<filter\>, if it 
	 * detects another comparation different from the first one in the same block, it throws this exception
	 * <p>
	 * Ex. WHERE a=1 AND b=2 OR c=3 throws an exception.
	 * <p>
	 * For solving this problem use agruppator
	 * <p>
	 * Ex. WHERE a=1 AND (b=2 OR c=3)
	 */
	public void and() throws DynamicsCrmQueryException {
		comparation(Comparation.AND);
	}

	/**
	 * Define the comparation of the \<filter\> 
	 * @throws DynamicsCrmQueryException By rule, the Dynamics CRM Query Language only allows one comparation per \<filter\>, if it 
	 * detects another comparation different from the first one in the same block, it throws this exception
	 * <p>
	 * Ex. WHERE a=1 AND b=2 OR c=3 throws an exception.
	 * <p>
	 * For solving this problem use agruppator
	 * <p>
	 * Ex. WHERE a=1 AND (b=2 OR c=3)
	 */
	public void or() {
		comparation(Comparation.OR);
	}

	/**
	 * Adds a comparison element that is represented by {@link DynamicsCrmWhereLeaf} and is translated as a \<condition\>
	 * 
	 * @param operator The operator for the condition
	 * @param field The name of the field
	 * @param value The value to compare to
	 */
	public void comparison(String operator, String field, String value) {
		lastComposite.addComponent(new DynamicsCrmWhereLeaf(operator, field, value));
	}
	
	private void comparation(Comparation comparationToAdd) {
		Comparation comparation = lastComposite.getComparation();
		// Check that always the same operator is beign used within the filter
		if (comparation != null && !comparation.equals(comparationToAdd)) {
			throw new DynamicsCrmQueryException("The Native Query Language of Dynamics CRM does not support two different logical operators (AND, OR) in the same block. For an operation like ... WHERE a=1 AND b=2 OR c=3 use instead aggrupators WHERE a=1 AND (b=2 OR c=3)");
		}
		
		lastComposite.setComparation(comparationToAdd);
	}

	@Override
	public String getQueryXml() {
		return firstComposite != null ? firstComposite.getQueryXml() : "";
	}

	@Override
	public void release() {
		if (firstComposite != null) {
			firstComposite.release();
		}
		firstComposite = null;
		lastComposite = null;
	}
}
