/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package org.mule.module.dynamicscrm.paging;

import static org.mule.module.dynamicscrm.paging.PagingHelper.getNativeQueryCurrentPageNumber;
import static org.mule.module.dynamicscrm.paging.PagingHelper.isNativeQueryCurrentPageNumber;
import static org.mule.module.dynamicscrm.paging.PagingHelper.isNativeQueryCurrentPageSize;
import static org.mule.module.dynamicscrm.paging.PagingHelper.setNativeQueryCurrentPageNumber;
import static org.mule.module.dynamicscrm.paging.PagingHelper.setNativeQueryCurrentPageSize;
import static org.mule.module.dynamicscrm.utils.DynamicsCrmUtils.mapEntityCollectionToMapCollection;

import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.api.MuleException;
import org.mule.module.dynamicscrm.paging.exception.DynamicsCrmPagingException;
import org.mule.streaming.PagingDelegate;

import com.microsoft.schemas.xrm._2011.contracts.EntityCollection;
import com.microsoft.schemas.xrm._2011.contracts.FetchExpression;
import com.microsoft.schemas.xrm._2011.contracts.services.IOrganizationService;
import com.microsoft.schemas.xrm._2011.contracts.services.IOrganizationServiceRetrieveMultipleOrganizationServiceFaultFaultFaultMessage;

public class RetrieveMultipleByQueryPagingDelegate extends PagingDelegate<Map<String, Object>> {
	
	static final private Log LOG = LogFactory.getLog(RetrieveMultipleByQueryPagingDelegate.class);
	
	private IOrganizationService client;
	private EntityCollection lastEntityCollection;
	private String lastQuery;
	private boolean firstPageRequest;
	private int itemsPerPage;
	
	public RetrieveMultipleByQueryPagingDelegate(IOrganizationService client, String query, int itemsPerPage) {
		super();
		firstPageRequest = true;
		this.client = client;
		lastQuery = query;
		this.itemsPerPage = itemsPerPage;
	}

	@Override
	public List<Map<String, Object>> getPage() {
		
		LOG.debug("Requesting new page");
		
		// Check if it is the first page requested (if true, don't alter the current page)
		if (firstPageRequest) {
			firstPageRequest = false;
		} else {
			// The query already has a page number
			if (isNativeQueryCurrentPageNumber(lastQuery)) {
				// Retrieve the current page, increment in one, and modify the query with the new page number
				int currentPageNumber = getNativeQueryCurrentPageNumber(lastQuery);
				currentPageNumber ++;
				lastQuery = setNativeQueryCurrentPageNumber(lastQuery, currentPageNumber);
				LOG.debug("Set page to " + currentPageNumber);
			} else {
				// If it does not have a page, we request the next one that is the second
				lastQuery = setNativeQueryCurrentPageNumber(lastQuery, 2);
				LOG.debug("Set page to 2");
			}
		}
		
		// Don't change the ammount of items per page if the query already has it
		// This is to prevent changing it when a native query was introduced instead of a DSQL Query
		if (!isNativeQueryCurrentPageSize(lastQuery)) {
			// Only 1 or grater number allowed as items per page
			if (itemsPerPage > 0) {
				lastQuery = setNativeQueryCurrentPageSize(lastQuery, itemsPerPage);
				LOG.debug("Set fetchSize (items per page) to " + itemsPerPage);
			}
		}
		
		LOG.debug("Native Query: " + lastQuery);
		
		// Use the modified query to retrieve the data
		FetchExpression query = new FetchExpression();
		query.setQuery(lastQuery);
		try {
			lastEntityCollection = client.retrieveMultiple(query);
		} catch (IOrganizationServiceRetrieveMultipleOrganizationServiceFaultFaultFaultMessage e) {
			throw new DynamicsCrmPagingException(e);
		}
		
		LOG.debug("Requested new page");
		
		List<Map<String,Object>> mapEntityCollectionToMapCollection = mapEntityCollectionToMapCollection(lastEntityCollection);
		
		return mapEntityCollectionToMapCollection == null || mapEntityCollectionToMapCollection.isEmpty() ? null : mapEntityCollectionToMapCollection;
	}
	
	@Override
	public void close() throws MuleException {
		client = null;
		lastEntityCollection = null;
		lastQuery = null;
	}
	
	@Override
	public int getTotalResults() {
		return -1;
	}
}
