/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;

import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.CLIENT_CREDENTIALS;
import static java.util.Arrays.asList;
import static org.apache.http.HttpStatus.SC_OK;
import static org.junit.rules.ExpectedException.none;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;

import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.test.runner.RunnerDelegateTo;

import java.util.Collection;
import java.util.Collections;

import io.qameta.allure.Description;
import net.smartam.leeloo.client.request.OAuthClientRequest;
import org.apache.commons.httpclient.methods.PostMethod;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runners.Parameterized;

@RunnerDelegateTo(Parameterized.class)
public class NoResourceOwnerSecurityProviderTestCase extends AbstractOAuth2ProviderModuleTestCase {

  private static final String AUTHORIZED_GRANT_TYPES = "authorizedGrantTypes";

  @Rule
  public ExpectedException expectedException = none();

  private String name;

  @Rule
  public SystemProperty authorizedGrantTypes;

  @Parameterized.Parameters()
  public static Collection<Object[]> data() {
    return asList(new Object[][] {
        {"NO_AUTHORIZED_GRANT_TYPES", new SystemProperty(AUTHORIZED_GRANT_TYPES, ""), true},
        {"ONLY_CLIENT_CREDENTIALS", new SystemProperty(AUTHORIZED_GRANT_TYPES, "CLIENT_CREDENTIALS"), false},
        {"ONLY_RO_GRANT_TYPE", new SystemProperty(AUTHORIZED_GRANT_TYPES, "RESOURCE_OWNER_PASSWORD_CREDENTIALS"), true},
        {"MORE_THAN_ONE_WITH_CLIENT_CREDENTIALS",
            new SystemProperty(AUTHORIZED_GRANT_TYPES, "RESOURCE_OWNER_PASSWORD_CREDENTIALS, CLIENT_CREDENTIALS"), true},
        {"MORE_THAN_ONE_NO_CLIENT_CREDENTIALS",
            new SystemProperty(AUTHORIZED_GRANT_TYPES, "RESOURCE_OWNER_PASSWORD_CREDENTIALS, AUTHORIZATION_CODE"), true}
    });
  }

  public NoResourceOwnerSecurityProviderTestCase(String name, SystemProperty authorizedGrantTypes, boolean shouldFail) {
    this.name = name;
    this.authorizedGrantTypes = authorizedGrantTypes;
    if (shouldFail) {
      expectedException.expect(InitialisationException.class);
      expectedException.expectMessage("A Resource Owner Security Provided should be configured");
    }
  }

  @Override
  protected String doGetConfigFile() {
    return "no-ro-security-provider-config.xml";
  }

  @Test
  @Description("There is no need for configuring a Resource Owner Security Provider if onle CLIENT_CREDENTILS grant type is supported.")
  public void noROSecurityProviderWorksIfOnlyClientCredentials() throws Exception {
    check();
  }

  private void check() throws Exception {
    client.getAuthorizedGrantTypes().add(CLIENT_CREDENTIALS);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setParameter("grant_type", "client_credentials")
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
  }


}
