/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;

import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BEARER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.CLIENT_CREDENTIALS;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.PASSWORD;
import static com.mulesoft.modules.oauth2.provider.api.client.ClientType.PUBLIC;
import static net.smartam.leeloo.common.message.types.GrantType.AUTHORIZATION_CODE;
import static net.smartam.leeloo.common.message.types.GrantType.NONE;
import static org.apache.commons.httpclient.HttpStatus.SC_BAD_REQUEST;
import static org.apache.commons.httpclient.HttpStatus.SC_METHOD_NOT_ALLOWED;
import static org.apache.commons.httpclient.HttpStatus.SC_MOVED_TEMPORARILY;
import static org.apache.commons.httpclient.HttpStatus.SC_OK;
import static org.apache.commons.httpclient.HttpStatus.SC_UNAUTHORIZED;
import static org.apache.commons.io.FileUtils.getFile;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.emptyArray;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.isEmptyString;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.UNAUTHORIZED;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.WWW_AUTHENTICATE;
import static org.mule.runtime.http.api.HttpHeaders.Values.APPLICATION_X_WWW_FORM_URLENCODED;
import static org.mule.runtime.api.metadata.MediaType.HTML;

import org.mule.runtime.core.api.util.FileUtils;
import org.mule.runtime.core.api.util.IOUtils;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.security.GeneralSecurityException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.net.ssl.SSLContext;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import net.smartam.leeloo.common.message.types.GrantType;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContexts;
import org.junit.Test;

public class OAuth2ProviderModuleCoreSecureTestCase extends AbstractOAuth2ProviderModuleTestCase {

  private static final String PROTECTED_RESOURCE_PATH = "/protected";
  private static final String CLIENT_RESTRICTED_RESOURCE_PATH = "/client_only";
  private static final String keyStorePath = "tls/clientKeystore";
  private static final String trustStorePath = "tls/trustStore";
  private static final String storePassword = "mulepassword";
  private static final String keyPassword = "mulepassword";
  private static final String protocol = "TLSv1.2";

  @Override
  protected String doGetConfigFile() {
    return "oauth2-core-tests-https-config.xml";
  }

  @Override
  protected String getProtocol() {
    return "https";
  }

  protected static HttpResponse executeGetRequest(String url) throws IOException, GeneralSecurityException {
    HttpClient client = getSecureClient(false);
    HttpGet getMethod = new HttpGet(url);
    return client.execute(getMethod);
  }

  private static HttpClient getSecureClient(boolean disableRedirectHandling) throws IOException, GeneralSecurityException {
    HttpClient secureClient;
    HttpClientBuilder clientBuilder = HttpClients.custom().setSslcontext(getSslContext());
    if (disableRedirectHandling) {
      clientBuilder.disableRedirectHandling();
    }
    secureClient = clientBuilder.build();
    return secureClient;
  }

  private static SSLContext getSslContext() throws IOException, GeneralSecurityException {
    SSLContext customSslContext;
    File keyStore = getFile(FileUtils.getResourcePath(keyStorePath, OAuth2ProviderModuleCoreSecureTestCase.class));
    File trustStore = getFile(FileUtils.getResourcePath(trustStorePath, OAuth2ProviderModuleCoreSecureTestCase.class));
    char[] storePass = storePassword.toCharArray();
    char[] keyPass = keyPassword.toCharArray();
    customSslContext =
        SSLContexts.custom()
            .useProtocol(protocol)
            .loadKeyMaterial(keyStore, storePass, keyPass)
            .loadTrustMaterial(trustStore, storePass)
            .build();
    return customSslContext;
  }


  @Test
  public void accessLoginPageBadMethod() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .buildQueryMessage();

    final HttpPut putAuthorization = new HttpPut(authorizationRequest.getLocationUri());
    HttpResponse response = getSecureClient(false).execute(putAuthorization);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_METHOD_NOT_ALLOWED));
  }

  @Test
  public void accessLoginPageEmptyRequest() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .buildQueryMessage();

    final HttpGet getAuthorization = new HttpGet(authorizationRequest.getLocationUri());
    HttpResponse response = getSecureClient(false).execute(getAuthorization);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    final String responseBody = IOUtils.toString(response.getEntity().getContent());
    final Map<String, List<String>> urlParameters = decodeParameters(responseBody);
    assertThat(urlParameters.get("error").get(0), is(equalTo("unsupported_response_type")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Missing mandatory parameter: response_type")));
  }

  @Test
  public void accessLoginPageBadResponseType() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("_bad_")
        .buildQueryMessage();

    final HttpGet getAuthorization = new HttpGet(authorizationRequest.getLocationUri());
    HttpResponse response = getSecureClient(false).execute(getAuthorization);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    final String responseBody = IOUtils.toString(response.getEntity().getContent());
    final Map<String, List<String>> urlParameters = decodeParameters(responseBody);
    assertThat(urlParameters.get("error").get(0), is(equalTo("unsupported_response_type")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Response type '_bad_' is not supported")));
  }

  @Test
  public void accessLoginPageBadClientId() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId("_bad_")
        .buildQueryMessage();

    final HttpGet getAuthorization = new HttpGet(authorizationRequest.getLocationUri());
    HttpResponse response = getSecureClient(false).execute(getAuthorization);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    final String responseBody = IOUtils.toString(response.getEntity().getContent());
    final Map<String, List<String>> urlParameters = decodeParameters(responseBody);
    assertThat(urlParameters.get("error").get(0), is(equalTo("unauthorized_client")));
  }

  @Test
  public void accessLoginPageBadRedirectUri() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI("_bad_")
        .buildQueryMessage();

    final HttpGet getAuthorization = new HttpGet(authorizationRequest.getLocationUri());
    HttpResponse response = getSecureClient(false).execute(getAuthorization);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    final String responseBody = IOUtils.toString(response.getEntity().getContent());
    final Map<String, List<String>> urlParameters = decodeParameters(responseBody);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_redirection_uri")));
  }

  @Test
  public void accessLoginPageWithScopeFailureAuthorizationCodeGrant() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setScope(TEST_SCOPE)
        .buildQueryMessage();

    final HttpGet getAuthorization = new HttpGet(authorizationRequest.getLocationUri());
    HttpResponse response = getSecureClient(true).execute(getAuthorization);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_MOVED_TEMPORARILY));

    final String location = response.getHeaders("Location")[0].getValue();
    assertThat(location, is(not(nullValue())));

    final URI locationUri = new URI(location);
    assertThat("authorization code grant type location has query", locationUri.getQuery(), is(not(nullValue())));

    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_scope")));
  }

  @Test
  public void accessLoginPageWithScopeFailureImplicitGrant() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("token")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setScope(TEST_SCOPE)
        .buildQueryMessage();

    final HttpGet getAuthorization = new HttpGet(authorizationRequest.getLocationUri());
    HttpResponse response = getSecureClient(true).execute(getAuthorization);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_MOVED_TEMPORARILY));

    final String location = response.getHeaders("Location")[0].getValue();
    assertThat(location, is(not(nullValue())));

    final URI locationUri = new URI(location);
    assertThat("token grant type location has no query", locationUri.getQuery(), is(nullValue()));
    assertThat("token grant type location has fragment", locationUri.getFragment(), is(not(nullValue())));

    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_scope")));
  }

  @Test
  public void accessLoginPageSuccess() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildQueryMessage();

    final HttpGet getAuthorization = new HttpGet(authorizationRequest.getLocationUri());
    HttpResponse response = getSecureClient(false).execute(getAuthorization);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));

    assertThat(response.getHeaders(CONTENT_TYPE)[0].getValue(), is(equalTo(HTML.toRfcString())));

    final String responseBody = IOUtils.toString(response.getEntity().getContent());
    assertThat(responseBody, containsString("<html>"));
    assertHasFormFieldContaining(responseBody, "code");
    assertHasFormFieldContaining(responseBody, TEST_CLIENT_ID);
    assertHasFormFieldContaining(responseBody, TEST_REDIRECT_URI);
  }

  @Test
  public void validateCredentialsNoParamProvided() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .buildBodyMessage();

    final HttpPost postCredentials = new HttpPost(authorizationRequest.getLocationUri());
    HttpResponse response = getSecureClient(false).execute(postCredentials);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    assertThat(response.getHeaders("Location"), emptyArray());
    final String responseBody = IOUtils.toString(response.getEntity().getContent());
    final Map<String, List<String>> urlParameters = decodeParameters(responseBody);
    assertThat(urlParameters.get("error").get(0), is(equalTo("unsupported_response_type")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Missing mandatory parameter: response_type")));
  }

  @Test
  public void validateCredentialsNoCredentialsProvided() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final HttpPost postCredentials = new HttpPost(authorizationRequest.getLocationUri());
    postCredentials.setEntity(new StringEntity(authorizationRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postCredentials);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_MOVED_TEMPORARILY));

    final String location = response.getHeaders("Location")[0].getValue();
    assertThat(location, is(not(nullValue())));

    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_request")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Missing mandatory parameter: username")));
    final String responseBody = IOUtils.toString(response.getEntity().getContent());
    assertThat(responseBody, isEmptyString());
  }

  @Test
  public void validateCredentialsInvalidCredentialsAuthorizationCodeGrant() throws Exception {
    doValidateCredentialsInvalidCredentials("code");
  }

  @Test
  public void validateCredentialsInvalidCredentialsImplicitGrant() throws Exception {
    doValidateCredentialsInvalidCredentials("token");
  }

  @Test
  public void validateCredentialsValidCredentials() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final HttpPost postCredentials = new HttpPost(authorizationRequest.getLocationUri());
    postCredentials.setEntity(new StringEntity(authorizationRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postCredentials);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_MOVED_TEMPORARILY));

    validateSuccessfulLoginResponse(response, "code");
  }

  @Test
  public void validateCredentialsValidCredentialsWithState() throws Exception {
    final String testState = RandomStringUtils.randomAlphanumeric(10);

    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setState(testState)
        .buildBodyMessage();

    final HttpPost postCredentials = new HttpPost(authorizationRequest.getLocationUri());
    postCredentials.setEntity(new StringEntity(authorizationRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postCredentials);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_MOVED_TEMPORARILY));

    final Map<String, List<String>> urlParameters = validateSuccessfulLoginResponse(response,
                                                                                    "code");
    assertThat(urlParameters.get("state").get(0), is(equalTo(testState)));
  }

  @Test
  public void tokenExchangeEmptyRequest() throws Exception {
    final HttpPost postToken = new HttpPost(getTokenEndpointURL());
    postToken.setHeader(CONTENT_TYPE, APPLICATION_X_WWW_FORM_URLENCODED.toRfcString());
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    assertEqualJsonObj(
                       "{\"error\":\"invalid_request\",\"error_description\":\"Missing mandatory parameter: grant_type\"}",
                       response);
  }

  @Test
  public void tokenExchangeUnsupportedGrantType() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(NONE)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(getTokenEndpointURL());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    assertEqualJsonObj(
                       "{\"error\":\"unsupported_grant_type\",\"error_description\":\"Grant type 'none' is not supported\"}",
                       response);
  }

  @Test
  public void tokenExchangeNoCredentials() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(getTokenEndpointURL());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    assertEqualJsonObj("{\"error\":\"invalid_client\",\"error_description\":\"Invalid credentials\"}",
                       response);
  }

  @Test
  public void tokenExchangeBrokenAuthorization() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, "_broken_");
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    assertEqualJsonObj(
                       "{\"error\":\"invalid_request\",\"error_description\":\"Invalid 'Authorization' header\"}",
                       response);
  }

  @Test
  public void tokenExchangeBadAuthorization() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, getValidBasicAuthHeaderValue("_bad_", "_bad_"));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_UNAUTHORIZED));

    assertThat(IOUtils.toString(response.getEntity().getContent()), containsString("\"error\":\"invalid_client\""));
  }

  @Test
  public void tokenExchangeInvalidRequestUri() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setCode("_ignored_")
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    assertEqualJsonObj(
                       "{\"error\":\"invalid_redirection_uri\",\"error_description\":\"Missing mandatory parameter: redirect_uri\"}",
                       response);
  }

  @Test
  public void tokenExchangeInvalidGrant() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setCode("_invalid_")
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    assertEqualJsonObj("{\"error\":\"invalid_grant\",\"error_description\":\"Authorization code is invalid or expired\"}",
                       response);
  }

  @Test
  public void tokenExchangeMultipleAuthentications() throws Exception {

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .buildBodyMessage();

    oAuthClientRequest.setHeaders(Collections.singletonMap(AUTHORIZATION,
                                                           getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    assertEqualJsonObj(
                       "{\"error\":\"invalid_request\",\"error_description\":\"Multiple client authentications found\"}",
                       response);
  }

  @Test
  public void tokenExchangeExpiredAuthorizationCode() throws Exception {
    tokenExchangeValidUsernamePassword(); //Run so that the authentication code is consumed
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    assertEqualJsonObj(
                       "{\"error\":\"invalid_grant\",\"error_description\":\"Authorization code is invalid or expired\"}",
                       response);
  }

  @Test
  public void tokenExchangeClientNotFoundInSecurityProvider() throws Exception {
    client.setPrincipal(null);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, "_bad_"));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_UNAUTHORIZED));

    assertThat(response.getHeaders(WWW_AUTHENTICATE)[0], is(not(nullValue())));

    assertEqualJsonObj("{\"error\":\"invalid_client\",\"error_description\":\"Invalid credentials\"}",
                       response);
  }

  @Test
  public void tokenExchangeClientIdIsValidSecurityProviderPrincipal() throws Exception {
    setupClient(TEST_CLIENT_OPTIONAL_PRINCIPAL, null);

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(response));
  }

  @Test
  public void tokenExchangeValidUsernamePassword() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(response));
  }

  @Test
  public void tokenExchangeValidBasicAuth() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(response));
  }

  @Test
  public void tokenExchangeValidClientSecret() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(response));
  }

  @Test
  public void tokenExchangePublicClient() throws Exception {
    client.setSecret(null);
    client.setType(PUBLIC);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(response));
  }

  @Test
  public void tokenRequestBadClientAuthorization() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.PASSWORD)
        .setUsername(TEST_RESOURCE_OWNER_USERNAME)
        .setPassword(TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, getValidBasicAuthHeaderValue("_bad_", "_bad_"));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_UNAUTHORIZED));

    assertThat(IOUtils.toString(response.getEntity().getContent()), containsString("\"error\":\"invalid_client\""));
  }

  @Test
  public void tokenRequestBadResourceOwnerCredentials() throws Exception {
    client.getAuthorizedGrantTypes().add(PASSWORD);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.PASSWORD)
        .setUsername(TEST_RESOURCE_OWNER_USERNAME)
        .setPassword("_bad_")
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_BAD_REQUEST));

    assertThat(IOUtils.toString(response.getEntity().getContent()), containsString("\"error\":\"access_denied\""));
  }

  @Test
  public void accessProtectedResourceWithoutToken() throws Exception {
    final HttpGet getProtectedResource = new HttpGet(getProtectedResourceURL(PROTECTED_RESOURCE_PATH));
    HttpResponse response = getSecureClient(false).execute(getProtectedResource);
    assertThat(response.getStatusLine().getStatusCode(), is(UNAUTHORIZED.getStatusCode()));
    assertThat(response.getHeaders(WWW_AUTHENTICATE)[0], is(not(nullValue())));
    assertThat(response.getHeaders(WWW_AUTHENTICATE)[0].getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }

  @Test
  public void accessProtectedResourceWithBadAccessToken() throws Exception {
    final HttpGet getProtectedResource = new HttpGet(getProtectedResourceURL(PROTECTED_RESOURCE_PATH)
        + "?access_token=_bad_");
    HttpResponse response = getSecureClient(false).execute(getProtectedResource);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_UNAUTHORIZED));

    assertThat(response.getHeaders(WWW_AUTHENTICATE)[0], is(not(nullValue())));
    assertThat(response.getHeaders(WWW_AUTHENTICATE)[0].getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }

  @Test
  public void accessProtectedResourceWithExpiredAccessToken() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);

    final HttpGet getProtectedResource = new HttpGet(getProtectedResourceURL(PROTECTED_RESOURCE_PATH)
        + "?access_token=" + accessToken);
    HttpResponse response = getSecureClient(false).execute(getProtectedResource);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_UNAUTHORIZED));

    assertThat(response.getHeaders(WWW_AUTHENTICATE)[0], is(not(nullValue())));
    assertThat(response.getHeaders(WWW_AUTHENTICATE)[0].getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }

  @Test
  public void accessProtectedResourceWithAccessTokenQueryParam() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);
    addAccessTokenToStore(accessToken);

    accessProtectedResource(accessToken);
  }

  @Test
  public void accessProtectedResourceWithBearerHeader() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);
    addAccessTokenToStore(accessToken);

    final HttpGet getProtectedResource = new HttpGet(getProtectedResourceURL(PROTECTED_RESOURCE_PATH + "-with-bearer"));
    getProtectedResource.addHeader("Authorization", "Bearer " + accessToken);
    HttpResponse response = getSecureClient(false).execute(getProtectedResource);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));

    assertThat(IOUtils.toString(response.getEntity().getContent()), is(equalTo(PROTECTED_RESOURCE_CONTENT)));
    assertThat(response.getHeaders(WWW_AUTHENTICATE), is(emptyArray()));
  }

  @Test
  public void performAuthorizationCodeOAuth2DanceAndAccessProtectedResource() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final HttpPost postCredentials = new HttpPost(authorizationRequest.getLocationUri());
    postCredentials.setEntity(new StringEntity(authorizationRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postCredentials);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_MOVED_TEMPORARILY));

    final Map<String, List<String>> authorizationResponse = validateSuccessfulLoginResponse(response, "code");
    final String authorizationCode = authorizationResponse.get("code").get(0);

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(authorizationCode)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(response));
    final String accessToken = (String) tokenResponse.get("access_token");

    accessProtectedResource(accessToken);
  }

  @Test
  public void performImplicitGrantOAuth2DanceAndAccessProtectedResource() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("token")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(authorizationRequest.getLocationUri());
    postToken.setEntity(new StringEntity(authorizationRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_MOVED_TEMPORARILY));

    final Map<String, List<String>> tokenResponse = validateSuccessfulLoginResponse(response, "access_token");
    final String accessToken = tokenResponse.get("access_token").get(0);

    accessProtectedResource(accessToken);
  }

  @Test
  public void performResourceOwnerPasswordCredentialsGrantOAuth2DanceAndAccessProtectedResource()
      throws Exception {
    client.getAuthorizedGrantTypes().add(PASSWORD);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.PASSWORD)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();


    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(response));
    final String accessToken = (String) tokenResponse.get("access_token");

    accessProtectedResource(accessToken);
  }

  @Test
  public void performClientCredentialsGrantOAuth2DanceAndAccessProtectedResource() throws Exception {
    client.getAuthorizedGrantTypes().add(CLIENT_CREDENTIALS);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setParameter("grant_type", "client_credentials")
        .buildBodyMessage();

    final HttpPost postToken = new HttpPost(oAuthClientRequest.getLocationUri());
    postToken.setEntity(new StringEntity(oAuthClientRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    postToken.setHeader(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD));
    HttpResponse response = getSecureClient(false).execute(postToken);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(response));
    final String accessToken = (String) tokenResponse.get("access_token");

    accessProtectedResource(accessToken);
  }

  @Override
  protected void accessProtectedResource(final String accessToken) throws Exception {
    final HttpGet getProtectedResource = new HttpGet(getProtectedResourceURL(PROTECTED_RESOURCE_PATH)
        + "?access_token=" + accessToken);
    HttpResponse response = getSecureClient(false).execute(getProtectedResource);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_OK));
    assertThat(response.getHeaders(WWW_AUTHENTICATE), is(emptyArray()));
    assertThat(IOUtils.toString(response.getEntity().getContent()), is(equalTo(PROTECTED_RESOURCE_CONTENT)));
  }

  private void doValidateCredentialsInvalidCredentials(final String responseType) throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType(responseType)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", "__BAD__")
        .buildBodyMessage();

    HttpPost httpPost = new HttpPost(authorizationRequest.getLocationUri());
    httpPost.setEntity(new StringEntity(authorizationRequest.getBody(), ContentType.APPLICATION_FORM_URLENCODED));
    HttpResponse response = getSecureClient(false).execute(httpPost);
    assertThat(response.getStatusLine().getStatusCode(), is(SC_MOVED_TEMPORARILY));

    final String location = response.getHeaders("Location")[0].getValue();
    assertThat(location, is(not(nullValue())));
    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("access_denied")));
    final String responseBody = IOUtils.toString(response.getEntity().getContent());
    assertThat(responseBody, isEmptyString());
  }
}
