/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;

import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BEARER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.AUTHORIZATION_CODE;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.CLIENT_CREDENTIALS;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.PASSWORD;
import static org.apache.http.HttpStatus.SC_OK;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.AllOf.allOf;
import static org.junit.rules.ExpectedException.none;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.FORBIDDEN;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.MOVED_TEMPORARILY;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.WWW_AUTHENTICATE;
import static org.mule.runtime.api.metadata.MediaType.HTML;

import com.mulesoft.modules.oauth2.provider.api.token.AccessTokenStoreHolder;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import net.smartam.leeloo.common.message.types.GrantType;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class OAuth2ProviderModuleDefaultScopeTestCase extends AbstractOAuth2ProviderModuleTestCase {

  @Rule
  public ExpectedException expectedException = none();

  @Override
  protected String doGetConfigFile() {
    return "oauth2-default-scope-http-config.xml";
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();

    client.getAuthorizedGrantTypes().add(AUTHORIZATION_CODE);
    updateClientInOS();
  }

  @Test
  public void accessLoginPageWithScopeSuccess() throws Exception {
    final String clientName = RandomStringUtils.randomAlphanumeric(20);
    client.setClientName(clientName);
    updateClientInOS();

    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setScope(USER_SCOPE)
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    executeHttpMethodExpectingStatus(getAuthorization, SC_OK);

    assertThat(getAuthorization.getResponseHeader(CONTENT_TYPE).getValue(), is(equalTo(HTML.toRfcString())));

    final String responseBody = getAuthorization.getResponseBodyAsString();
    assertThat(responseBody, allOf(
                                   containsString("<html>"),
                                   containsString("<li>" + USER_SCOPE + "</li>"),
                                   containsString(clientName),
                                   containsString("TestScopedProvider")));
    assertHasFormFieldContaining(responseBody, "code");
    assertHasFormFieldContaining(responseBody, TEST_CLIENT_ID);
    assertHasFormFieldContaining(responseBody, TEST_REDIRECT_URI);
    assertHasFormFieldContaining(responseBody, USER_SCOPE);
  }

  @Test
  public void validateCredentialsSuccess() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setScope(USER_SCOPE)
        .buildBodyMessage();

    final PostMethod postCredentials =
        postOAuthClientRequestExpectingStatus(authorizationRequest, MOVED_TEMPORARILY.getStatusCode());

    validateSuccessfulLoginResponse(postCredentials, "code");
  }

  @Test
  public void tokenExchangeSuccess() throws Exception {
    authorizationCodeStoreHolder.getAuthorizationRequest().getScopes().add(USER_SCOPE);
    updateAuthorizationCodeInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest,
                                                                       SC_OK);

    validateSuccessfulTokenResponseNoRefresh(getContentAsMap(postToken), USER_SCOPE);
  }

  @Test
  public void accessScopelessProtectedResourceSuccess() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);
    addAccessTokenToStore(accessToken);

    accessScopelessProtectedResource(accessToken);
  }

  @Test
  public void accessScopedProtectedResourceWithBadScope() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);
    addAccessTokenToStore(accessToken);

    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL("/protected/scopes")
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, FORBIDDEN.getStatusCode());
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE), is(not(nullValue())));
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE).getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }

  @Test
  public void accessScopedProtectedResourceSuccess() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);
    final AccessTokenStoreHolder accessTokenStoreHolder = addAccessTokenToStore(accessToken);
    accessTokenStoreHolder.getAccessToken().getScopes().add(USER_SCOPE);
    updateAccessTokenHolderInOS(accessTokenStoreHolder);

    accessScopedProtectedResource(accessToken);
  }

  @Test
  public void performAuthorizationCodeOAuth2DanceAndAccessProtectedResources() throws Exception {
    final String authorizationCode = getAuthorizationCode();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.AUTHORIZATION_CODE)
        .setCode(authorizationCode)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest,
                                                                       SC_OK);

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoRefresh(getContentAsMap(postToken),
                                                                                       USER_SCOPE);
    final String accessToken = (String) tokenResponse.get("access_token");

    accessScopelessProtectedResource(accessToken);
    accessScopedProtectedResource(accessToken);
    accessForbiddenScopedProtectedResource(accessToken);
  }

  private String getAuthorizationCode(String clientId, String scope) throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(clientId)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setScope(scope)
        .buildBodyMessage();

    final PostMethod postCredentials = postOAuthClientRequestExpectingStatus(authorizationRequest,
                                                                             MOVED_TEMPORARILY.getStatusCode());

    final Map<String, List<String>> authorizationResponse = validateSuccessfulLoginResponse(
                                                                                            postCredentials, "code");
    return authorizationResponse.get("code").get(0);
  }

  private String getAuthorizationCode() throws Exception {
    return getAuthorizationCode(TEST_CLIENT_ID, USER_SCOPE);
  }


  @Test
  public void performImplicitGrantOAuth2DanceAndAccessProtectedResource() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("token")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setScope(USER_SCOPE)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(authorizationRequest,
                                                                       MOVED_TEMPORARILY.getStatusCode());

    final Map<String, List<String>> tokenResponse = validateSuccessfulLoginResponse(postToken,
                                                                                    "access_token");
    final String accessToken = tokenResponse.get("access_token").get(0);

    accessScopelessProtectedResource(accessToken);
    accessScopedProtectedResource(accessToken);
    accessForbiddenScopedProtectedResource(accessToken);
  }

  @Test
  public void performResourceOwnerPasswordCredentialsGrantOAuth2DanceAndAccessProtectedResource()
      throws Exception {
    client.getAuthorizedGrantTypes().add(PASSWORD);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.PASSWORD)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setScope(USER_SCOPE)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest,
                                                                       SC_OK);

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoRefresh(getContentAsMap(postToken),
                                                                                       USER_SCOPE);
    final String accessToken = (String) tokenResponse.get("access_token");

    accessScopelessProtectedResource(accessToken);
    accessScopedProtectedResource(accessToken);
    accessForbiddenScopedProtectedResource(accessToken);
  }

  @Test
  public void performClientCredentialsGrantOAuth2DanceAndAccessProtectedResource() throws Exception {
    client.getAuthorizedGrantTypes().add(CLIENT_CREDENTIALS);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setParameter("grant_type", "client_credentials")
        .setScope(USER_SCOPE)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest,
                                                                       SC_OK);

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoRefresh(getContentAsMap(postToken),
                                                                                       USER_SCOPE);
    final String accessToken = (String) tokenResponse.get("access_token");

    accessScopelessProtectedResource(accessToken);
    accessScopedProtectedResource(accessToken);
    accessForbiddenScopedProtectedResource(accessToken);
    accessForbiddenScopedProtectedResourceSuccess(accessToken);
  }

  @Test
  public void getAuthorizationCodeWithDefaultScopeOnCreatedClient() throws Exception {
    getAuthorizationCode("customClient", USER_SCOPE);
    expectedException.expect(AssertionError.class);
    getAuthorizationCode("customClient", ADMIN_SCOPE);
  }

  private void accessScopelessProtectedResource(final String accessToken) throws IOException {
    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL("/protected/no-scope")
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, SC_OK);
    assertThat(getProtectedResource.getResponseBodyAsString(), is(equalTo(PROTECTED_RESOURCE_CONTENT)));
  }

  private void accessScopedProtectedResource(final String accessToken) throws IOException {
    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL("/protected/scopes")
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, SC_OK);
    assertThat(getProtectedResource.getResponseBodyAsString(), is(equalTo(PROTECTED_RESOURCE_CONTENT)));
  }

  private void accessForbiddenScopedProtectedResource(final String accessToken) throws IOException {
    final GetMethod getProtectedResource = new GetMethod(
                                                         getProtectedResourceURL("/protected/forbidden-scope") + "?access_token="
                                                             + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, FORBIDDEN.getStatusCode());
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE), is(not(nullValue())));
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE).getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }

  private void accessForbiddenScopedProtectedResourceSuccess(final String accessToken) throws IOException {
    client.getScopes().add(ADMIN_SCOPE);
    updateClientInOS();
    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL("/protected/scopes")
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, SC_OK);
    assertThat(getProtectedResource.getResponseBodyAsString(), is(equalTo(PROTECTED_RESOURCE_CONTENT)));
  }

}
