/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;

import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.AUTHORIZATION_CODE;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.REFRESH_TOKEN;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.BAD_REQUEST;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.MOVED_TEMPORARILY;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.UNAUTHORIZED;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import net.smartam.leeloo.common.exception.OAuthSystemException;
import net.smartam.leeloo.common.message.types.GrantType;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;

import org.junit.Test;

public class OAuth2ProviderModuleRefreshTokenWithoutIssuingNewTestCase extends AbstractOAuth2ProviderModuleTestCase {

  private static final String PROTECTED_RESOURCE_PATH = "/protected";

  @Override
  protected String doGetConfigFile() {
    return "oauth2-refresh-token-without-issuing-new-tests-config.xml";
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    client.getAuthorizedGrantTypes().add(AUTHORIZATION_CODE);
    client.getAuthorizedGrantTypes().add(REFRESH_TOKEN);
    updateClientInOS();
  }

  @Test
  public void performAuthorizationCodeGrantOAuth2DanceAndTestRefreshToken() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final PostMethod postCredentials =
        postOAuthClientRequestExpectingStatus(authorizationRequest, MOVED_TEMPORARILY.getStatusCode());

    final Map<String, List<String>> authorizationResponse = validateSuccessfulLoginResponse(
                                                                                            postCredentials, "code");
    final String authorizationCode = authorizationResponse.get("code").get(0);

    final OAuthClientRequest tokenExchangeRequest = OAuthClientRequest.tokenLocation(
                                                                                     getTokenEndpointURL())
        .setGrantType(GrantType.AUTHORIZATION_CODE)
        .setCode(authorizationCode)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    doGetAccessTokenAndTryRefreshIt(tokenExchangeRequest);
  }

  private void doGetAccessTokenAndTryRefreshIt(final OAuthClientRequest tokenExchangeRequest)
      throws IOException, OAuthSystemException, InterruptedException {
    PostMethod postToken = postOAuthClientRequestExpectingStatus(tokenExchangeRequest, OK.getStatusCode());

    Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScope(getContentAsMap(postToken), true);
    final String accessToken1 = (String) tokenResponse.get("access_token");
    final String refreshToken1 = (String) tokenResponse.get("refresh_token");

    GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL(PROTECTED_RESOURCE_PATH)
        + "?access_token=" + accessToken1);
    executeHttpMethodExpectingStatus(getProtectedResource, OK.getStatusCode());
    assertThat(PROTECTED_RESOURCE_CONTENT, equalTo(getProtectedResource.getResponseBodyAsString()));

    OAuthClientRequest refreshTokenRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.REFRESH_TOKEN)
        .setRefreshToken(refreshToken1)
        .buildBodyMessage();
    refreshTokenRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    postToken = postOAuthClientRequestExpectingStatus(refreshTokenRequest, OK.getStatusCode());

    tokenResponse = validateSuccessfulTokenResponseNoScope(getContentAsMap(postToken), true);
    final String accessToken2 = (String) tokenResponse.get("access_token");
    final String refreshToken2 = (String) tokenResponse.get("refresh_token");

    assertThat(accessToken2, not(equalTo(accessToken1)));
    assertThat(refreshToken2, equalTo(refreshToken1));

    getProtectedResource = new GetMethod(getProtectedResourceURL(PROTECTED_RESOURCE_PATH)
        + "?access_token=" + accessToken2);
    executeHttpMethodExpectingStatus(getProtectedResource, OK.getStatusCode());
    assertThat(PROTECTED_RESOURCE_CONTENT, equalTo(getProtectedResource.getResponseBodyAsString()));

    // Make sure the refresh token is expired
    Thread.sleep(5000);

    refreshTokenRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.REFRESH_TOKEN)
        .setRefreshToken(refreshToken1)
        .buildBodyMessage();
    refreshTokenRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    postOAuthClientRequestExpectingStatus(refreshTokenRequest, BAD_REQUEST.getStatusCode());

    // Make sure the access tokens granted before are not valid either
    getProtectedResource = new GetMethod(getProtectedResourceURL(PROTECTED_RESOURCE_PATH)
        + "?access_token=" + accessToken1);
    executeHttpMethodExpectingStatus(getProtectedResource, UNAUTHORIZED.getStatusCode());

    getProtectedResource = new GetMethod(getProtectedResourceURL(PROTECTED_RESOURCE_PATH)
        + "?access_token=" + accessToken2);
    executeHttpMethodExpectingStatus(getProtectedResource, UNAUTHORIZED.getStatusCode());
  }
}
