/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;

import static java.lang.String.format;
import static java.util.Collections.singletonMap;
import static net.smartam.leeloo.client.request.OAuthClientRequest.tokenLocation;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.core.IsNull.nullValue;
import static org.junit.Assert.fail;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.MOVED_TEMPORARILY;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreManager;

import com.mulesoft.modules.oauth2.provider.api.client.Client;
import com.mulesoft.modules.oauth2.provider.api.client.ObjectStoreClientStore;
import com.mulesoft.modules.oauth2.provider.api.code.ObjectStoreAuthorizationCode;
import com.mulesoft.modules.oauth2.provider.api.token.AccessTokenStoreHolder;

import java.util.List;
import java.util.Map;

import javax.inject.Inject;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import net.smartam.leeloo.common.message.types.GrantType;
import org.apache.commons.httpclient.methods.PostMethod;
import org.junit.Before;
import org.junit.Test;

public class PrivateObjectStoreTestCase extends AbstractOAuth2ProviderModuleTestCase {

  private static final String PRIVATE_CLIENT_STORE_OBJECT_STORE = "clientsPrivateObjectStore";
  private static final String PRIVATE_TOKENS_STORE_OBJECT_STORE = "tokensPrivateObjectStore";
  private static final String PRIVATE_AUTHORIZATION_CODE_STORE_OBJECT_STORE = "authorizationPrivateObjectStore";

  @Before
  public void initializeStores() throws Exception {
    ObjectStore privateClientObjectStore = objectStoreManager.getObjectStore(PRIVATE_CLIENT_STORE_OBJECT_STORE);

    ((ObjectStoreClientStore) clientStore).setObjectStore(privateClientObjectStore);
    initializeClientObjectStore();

    ObjectStore privateAuthorizationObjectStore =
        objectStoreManager.getObjectStore(PRIVATE_AUTHORIZATION_CODE_STORE_OBJECT_STORE);

    ((ObjectStoreAuthorizationCode) authorizationCodeStore).setObjectStore(privateAuthorizationObjectStore);
    initializeAuthorizationCodeObjectStore();
  }

  @Inject
  private ObjectStoreManager objectStoreManager;

  @Override
  protected String doGetConfigFile() {
    return "oauth2-private-objectstore-client-store-config.xml";
  }

  @Test
  public void checkPrivateObjectStoreForTokenIsWorking() throws Exception {
    final OAuthClientRequest oAuthClientRequest = tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    oAuthClientRequest.setHeaders(singletonMap(AUTHORIZATION,
                                               getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, OK.getStatusCode());

    Map<String, Object> parameters = validateSuccessfulTokenResponseNoScope(getContentAsMap(postToken), false);
    String accessToken = (String) parameters.get("access_token");

    ObjectStore tokenStore = objectStoreManager.getObjectStore(PRIVATE_TOKENS_STORE_OBJECT_STORE);
    Map<String, AccessTokenStoreHolder> tokens = tokenStore.retrieveAll();
    assertTokenInStore(tokens, accessToken);
  }

  private void assertTokenInStore(Map<String, AccessTokenStoreHolder> tokens, String expectedToken) throws Exception {
    for (Map.Entry<String, AccessTokenStoreHolder> entry : tokens.entrySet()) {
      if (entry.getValue().getAccessToken().getAccessToken().equals(expectedToken)) {
        return;
      }
    }
    fail(format("%s not in saved tokens", expectedToken));
  }

  @Test
  public void checkPrivateObjectStoreForClientStoreIsWorking() throws Exception {
    final String flowName = "registerClient";
    flowRunner(flowName).run();

    ObjectStore privateClientObjectStore = objectStoreManager.getObjectStore("clientsPrivateObjectStore");
    Map<String, Client> clients = privateClientObjectStore.retrieveAll();
    assertClientIdInStore(clients, "TheClientId");
  }

  private void assertClientIdInStore(Map<String, Client> clients, String expectedId) throws Exception {
    for (Map.Entry<String, Client> entry : clients.entrySet()) {
      if (entry.getValue().getClientId().equals(expectedId)) {
        return;
      }
    }
    fail(format("%s not in saved clients", expectedId));
  }

  @Test
  public void checkPrivateObjectStoreForAuthorizationCodeIsWorking() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final PostMethod postCredentials =
        postOAuthClientRequestExpectingStatus(authorizationRequest, MOVED_TEMPORARILY.getStatusCode());

    Map<String, List<String>> responseQueryParams = decodeParameters(postCredentials.getResponseHeader("Location").toString());

    String code = responseQueryParams.get("code").get(0).trim();

    assertThat(authorizationCodeStore.retrieve(code), is(not(nullValue())));
  }
}
