/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.error;


import static java.lang.String.format;
import static java.util.Collections.singletonList;
import static net.smartam.leeloo.client.request.OAuthClientRequest.authorizationLocation;
import static net.smartam.leeloo.client.request.OAuthClientRequest.tokenLocation;
import static org.apache.http.HttpStatus.SC_BAD_REQUEST;
import static org.apache.http.HttpStatus.SC_MOVED_TEMPORARILY;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.TOO_MANY_REQUESTS;
import static org.mule.runtime.http.api.HttpHeaders.Values.APPLICATION_X_WWW_FORM_URLENCODED;

import com.mulesoft.modules.oauth2.provider.AbstractOAuth2ProviderModuleTestCase;

import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import net.smartam.leeloo.client.request.OAuthClientRequest.TokenRequestBuilder;
import net.smartam.leeloo.client.request.OAuthClientRequest.AuthenticationRequestBuilder;
import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.methods.GetMethod;

import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.junit.Before;
import org.junit.Test;


public class OAuth2ProviderModuleAuthorizationRequestErrorsTestCase extends AbstractOAuth2ProviderModuleTestCase {

  private AuthenticationRequestBuilder authenticationRequestBuilder;
  private TokenRequestBuilder tokenRequestBuilder;

  @Before
  public void resetRequest() throws Exception {
    authenticationRequestBuilder = authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setScope("USER")
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD);

    tokenRequestBuilder = tokenLocation(getTokenEndpointURL());
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    client.setScopes(new HashSet<>(singletonList(USER_SCOPE)));
    updateClientInOS();
  }


  @Override
  protected String doGetConfigFile() {
    return "authorization-errors-config.xml";
  }

  private void assertFailingRequest(HttpMethodBase method, String expectedErrorCode, int statusCode) throws Exception {
    executeHttpMethodExpectingStatus(method, statusCode);
    String responseBody = method.getResponseBodyAsString();
    assertThat(responseBody, containsString(format("error=%s", expectedErrorCode)));
  }

  private void assertInvalidRequest(HttpMethodBase method, String expectedErrorCode) throws Exception {
    executeHttpMethodExpectingStatus(method, SC_MOVED_TEMPORARILY);
    String location = method.getResponseHeader("Location").getValue();
    Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo(expectedErrorCode)));
  }

  private void assertBadRequest(HttpMethodBase method, String expectedErrorCode) throws Exception {
    assertFailingRequest(method, expectedErrorCode, SC_BAD_REQUEST);
  }


  @Test
  public void duplicateGrantTypeWithGET() throws Exception {
    assertDuplicateParameterWithGET("response_type=some_other_response_type");
  }

  @Test
  public void duplicateClientIdTypeWithGET() throws Exception {
    assertDuplicateParameterWithGET("client_id=some_other_client_id");
  }

  @Test
  public void duplicateRedirectUriWithGET() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setParameter("redirect_uri", TEST_REDIRECT_URI + "/otherThing")
        .buildQueryMessage();
    GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    assertBadRequest(getAuthorization, "invalid_redirection_uri");
  }

  @Test
  public void duplicateScopeTypeWithGET() throws Exception {
    assertDuplicateParameterWithGET("scope=some_other_scope");
  }

  @Test
  public void missingResponseTypeWithGET() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setResponseType(null)
        .buildQueryMessage();
    GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    assertInvalidRequest(getAuthorization, "unsupported_response_type");
  }

  @Test
  public void missingClientIdWithGET() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setClientId(null)
        .buildQueryMessage();
    GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    assertInvalidRequest(getAuthorization, "invalid_request");
  }

  @Test
  public void invalidRedirectUriWithGET() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder.setRedirectURI("__bad__").buildQueryMessage();
    GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    assertBadRequest(getAuthorization, "invalid_redirection_uri");
  }

  @Test
  public void unauthorizedClientMissingClientWithGET() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setClientId("some_unauthorized_client")
        .buildQueryMessage();
    GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    assertInvalidRequest(getAuthorization, "unauthorized_client");
  }

  @Test
  public void unauthorizedClientWrongScopeWithGET() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setScope("wrong_scope")
        .buildQueryMessage();
    GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    assertInvalidRequest(getAuthorization, "invalid_scope");
  }

  @Test
  public void unsupportedResponseType() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setResponseType("some_unsupported_response_type")
        .buildQueryMessage();
    GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    assertInvalidRequest(getAuthorization, "unsupported_response_type");
  }

  @Test
  public void invalidScope() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setScope("some_invalid_scope")
        .buildQueryMessage();
    GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    assertInvalidRequest(getAuthorization, "invalid_scope");
  }

  @Test
  public void duplicateGrantTypeWithPOST() throws Exception {
    assertDuplicateParameterWithPOST("response_type=some_other_response_type");
  }


  @Test
  public void duplicateClientIdTypeWithPOST() throws Exception {
    assertDuplicateParameterWithPOST("client_id=some_other_client_id");
  }

  @Test
  public void duplicateRedirectUriWithPOST() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setParameter("redirect_uri", TEST_REDIRECT_URI + "/otherThing")
        .buildBodyMessage();
    PostMethod postAuthorization = buildPostMethod(authorizationRequest);
    assertBadRequest(postAuthorization, "invalid_redirection_uri");
  }

  @Test
  public void duplicateScopeTypeWithPOST() throws Exception {
    assertDuplicateParameterWithGET("scope=some_other_scope");
  }


  @Test
  public void missingResponseTypeWithPOST() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setResponseType(null)
        .buildBodyMessage();
    PostMethod postAuthorization = buildPostMethod(authorizationRequest);
    assertInvalidRequest(postAuthorization, "unsupported_response_type");
  }

  @Test
  public void missingClientIdWithPOST() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setClientId(null)
        .buildBodyMessage();
    PostMethod postAuthorization = buildPostMethod(authorizationRequest);
    assertInvalidRequest(postAuthorization, "invalid_request");
  }

  @Test
  public void invalidRedirectUriWithPOST() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder.setRedirectURI("__bad__").buildBodyMessage();
    PostMethod postAuthorization = buildPostMethod(authorizationRequest);
    assertBadRequest(postAuthorization, "invalid_redirection_uri");
  }

  @Test
  public void unauthorizedClientMissingClientWithPOST() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setClientId("some_unauthorized_client")
        .buildBodyMessage();
    PostMethod postAuthorization = buildPostMethod(authorizationRequest);
    assertInvalidRequest(postAuthorization, "unauthorized_client");
  }

  @Test
  public void unauthorizedClientWrongScopeWithPOST() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setScope("wrong_scope")
        .buildBodyMessage();
    PostMethod postAuthorization = buildPostMethod(authorizationRequest);
    assertInvalidRequest(postAuthorization, "invalid_scope");
  }

  @Test
  public void unsupportedResponseTypeWithPOST() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setResponseType("some_unsupported_response_type")
        .buildBodyMessage();
    PostMethod postAuthorization = buildPostMethod(authorizationRequest);
    assertInvalidRequest(postAuthorization, "unsupported_response_type");
  }

  @Test
  public void invalidScopeWithPOST() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setScope("some_invalid_scope")
        .buildBodyMessage();
    PostMethod postAuthorization = buildPostMethod(authorizationRequest);
    assertInvalidRequest(postAuthorization, "invalid_scope");
  }



  @Test
  public void tooManyRequests() throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder
        .setParameter("password", "__bad_password__")
        .buildBodyMessage();

    PostMethod post = buildPostMethod(authorizationRequest);



    assertInvalidRequest(post, "access_denied");
    assertInvalidRequest(post, "access_denied");

    executeHttpMethodExpectingStatus(post, TOO_MANY_REQUESTS.getStatusCode());
    executeHttpMethodExpectingStatus(post, TOO_MANY_REQUESTS.getStatusCode());
  }

  private void assertDuplicateParameterWithPOST(String extraParameter) throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder.buildBodyMessage();
    authorizationRequest.setBody(authorizationRequest.getBody() + "&" + extraParameter);
    PostMethod post = buildPostMethod(authorizationRequest);
    assertInvalidRequest(post, "invalid_request");
  }

  private void assertDuplicateParameterWithGET(String extraParameter) throws Exception {
    OAuthClientRequest authorizationRequest = authenticationRequestBuilder.buildQueryMessage();
    GetMethod get = new GetMethod(authorizationRequest.getLocationUri());
    get.setQueryString(get.getQueryString() + "&" + extraParameter);
    get.setFollowRedirects(false);
    assertInvalidRequest(get, "invalid_request");
  }

  private PostMethod buildPostMethod(OAuthClientRequest oAuthClientRequest) throws UnsupportedEncodingException {
    PostMethod postMethod = new PostMethod(oAuthClientRequest.getLocationUri());
    postMethod.setRequestEntity(new StringRequestEntity(oAuthClientRequest.getBody(),
                                                        APPLICATION_X_WWW_FORM_URLENCODED.toRfcString(),
                                                        Charset.defaultCharset().toString()));
    return postMethod;
  }

}
