/*
 * (c) 2003-2019 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;

import static com.mulesoft.modules.oauth2.provider.api.Constants.ACCESS_TOKEN_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.CODE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.AUTHORIZATION_CODE;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ResponseType.CODE;
import static com.mulesoft.modules.oauth2.provider.api.Constants.UTF_8;
import static com.mulesoft.modules.oauth2.provider.api.client.ClientType.CONFIDENTIAL;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.apache.http.HttpStatus.SC_OK;
import static org.apache.http.protocol.HTTP.CONTENT_TYPE;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.both;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasKey;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.collection.IsEmptyCollection.empty;
import static org.hamcrest.core.IsNull.notNullValue;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.StringStartsWith.startsWith;
import static org.mule.runtime.core.api.util.Base64.encodeBytes;
import static org.mule.runtime.extension.api.annotation.param.MediaType.APPLICATION_JSON;
import static org.mule.runtime.http.api.HttpHeaders.Names.WWW_AUTHENTICATE;
import static org.mule.runtime.http.api.HttpHeaders.Values.APPLICATION_X_WWW_FORM_URLENCODED;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.security.DefaultMuleCredentials;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.mulesoft.modules.oauth2.provider.api.client.Client;
import com.mulesoft.modules.oauth2.provider.api.client.ClientStore;
import com.mulesoft.modules.oauth2.provider.api.client.ObjectStoreClientStore;
import com.mulesoft.modules.oauth2.provider.api.code.AuthorizationCodeStore;
import com.mulesoft.modules.oauth2.provider.api.code.ObjectStoreAuthorizationCode;
import com.mulesoft.modules.oauth2.provider.api.token.ObjectStoreAccessAndRefreshTokenStore;
import com.mulesoft.modules.oauth2.provider.api.token.TokenStore;
import com.mulesoft.modules.oauth2.provider.api.AuthorizationRequest;
import com.mulesoft.modules.oauth2.provider.api.exception.OAuth2Exception;
import com.mulesoft.modules.oauth2.provider.api.code.AuthorizationCodeStoreHolder;
import com.mulesoft.modules.oauth2.provider.api.token.AccessTokenStoreHolder;
import com.mulesoft.modules.oauth2.provider.api.token.Token;
import com.mulesoft.modules.oauth2.provider.api.ResourceOwnerAuthentication;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.inject.Inject;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;

import org.hamcrest.Matchers;
import org.junit.Rule;

public abstract class AbstractOAuth2ProviderModuleTestCase extends MuleArtifactFunctionalTestCase {

  protected static final String TEST_CLIENT_OPTIONAL_PRINCIPAL = "clusr";
  public static final String TEST_REDIRECT_URI = "http://fake/redirect";
  protected static final String TEST_AUTHORIZATION_CODE = "__valid__";
  protected static final String TEST_RESOURCE_OWNER_USERNAME = "rousr";
  protected static final String TEST_RESOURCE_OWNER_PASSWORD = "ropwd+%";
  protected static final String TEST_CLIENT_ID = "clientId1";
  protected static final String TEST_CLIENT_SECRET = "clientSecret1";
  protected static final String TEST_CLIENT_PASSWORD = "clpwd+%";
  protected static final String TEST_SCOPE = "test_scope";
  protected static final String PROTECTED_RESOURCE_CONTENT = "accessing::protected_resource";
  protected static final String PROTECTED_RESOURCE_PATH = "/protected";

  protected static final String USER_SCOPE = "USER";
  protected static final String ADMIN_SCOPE = "ADMIN";

  private static final String DEFAULT_CLIENT_OBJECT_STORE_NAME = "clientObjectStore";
  private static final String DEFAULT_ACCESS_TOKEN_OBJECT_STORE_NAME = "tokenObjectStore";
  private static final String DEFAULT_REFRESH_TOKEN_OBJECT_STORE_NAME = "refreshTokenObjectStore";
  private static final String DEFAULT_AUTHORIZATION_OBJECT_STORE = "authorizationCodeObjectStore";

  @Rule
  public DynamicPort port = new DynamicPort("port");

  @Rule
  public SystemProperty mUnitDisableInitialStateManagerProperty =
      new SystemProperty("munit.disable.initial.state.manager", "true");

  @Inject
  protected Registry registry;

  protected ClientStore clientStore;

  protected AuthorizationCodeStore authorizationCodeStore;

  protected TokenStore tokenStore;

  protected HttpClient httpClient;
  protected Client client;
  protected AuthorizationCodeStoreHolder authorizationCodeStoreHolder;

  protected String getProtocol() {
    return "http";
  }

  protected String getCommonConfigFile() {
    return "common-config.xml";
  }

  protected abstract String doGetConfigFile();

  @Override
  protected String[] getConfigFiles() {
    return new String[] {getCommonConfigFile(), doGetConfigFile()};
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    httpClient = new HttpClient();

    clientStore = new ObjectStoreClientStore();
    ((ObjectStoreClientStore) clientStore)
        .setObjectStore(registry.<ObjectStore>lookupByName(DEFAULT_CLIENT_OBJECT_STORE_NAME).get());

    tokenStore = new ObjectStoreAccessAndRefreshTokenStore();
    ((ObjectStoreAccessAndRefreshTokenStore) tokenStore)
        .setAccessTokenObjectStore(registry.<ObjectStore>lookupByName(DEFAULT_ACCESS_TOKEN_OBJECT_STORE_NAME).get());
    ((ObjectStoreAccessAndRefreshTokenStore) tokenStore)
        .setRefreshTokenObjectStore(registry.<ObjectStore>lookupByName(DEFAULT_REFRESH_TOKEN_OBJECT_STORE_NAME).get());

    authorizationCodeStore = new ObjectStoreAuthorizationCode();
    ((ObjectStoreAuthorizationCode) authorizationCodeStore)
        .setObjectStore(registry.<ObjectStore>lookupByName(DEFAULT_AUTHORIZATION_OBJECT_STORE).get());

    initializeClientObjectStore();
    initializeAuthorizationCodeObjectStore();
  }

  protected void initializeClientObjectStore() throws OAuth2Exception {
    setupClient(TEST_CLIENT_ID, TEST_CLIENT_OPTIONAL_PRINCIPAL);
  }

  protected void setupClient(final String id, final String principal) throws OAuth2Exception {
    client = new Client(id, TEST_CLIENT_SECRET, CONFIDENTIAL, null, null, null);
    client.getAuthorizedGrantTypes().add(AUTHORIZATION_CODE);
    client.getRedirectUris().add(TEST_REDIRECT_URI);
    client.setPrincipal(principal);
    clientStore.addClient(client, false);
  }

  protected Map<String, Object> getContentAsMap(HttpResponse response) throws IOException {
    return new Gson().fromJson(IOUtils.toString(response.getEntity().getContent()),
                               new TypeToken<Map<String, Object>>() {}.getType());
  }

  protected Map<String, Object> getContentAsMap(HttpMethod method) throws IOException {
    return new Gson().fromJson(method.getResponseBodyAsString(), new TypeToken<Map<String, Object>>() {}.getType());
  }

  protected void initializeAuthorizationCodeObjectStore() throws OAuth2Exception {
    createTestAuthorizationCode();
  }

  protected void createTestAuthorizationCode()
      throws OAuth2Exception {
    final AuthorizationRequest authorizationRequest = new AuthorizationRequest(TEST_CLIENT_ID, CODE, TEST_REDIRECT_URI,
                                                                               getResourceOwnerAuthentication());
    authorizationCodeStoreHolder = new AuthorizationCodeStoreHolder(TEST_AUTHORIZATION_CODE,
                                                                    authorizationRequest);
    authorizationCodeStore.store(authorizationCodeStoreHolder);
  }

  protected void updateAuthorizationCodeInOS() {
    authorizationCodeStore.store(authorizationCodeStoreHolder);
  }

  private ResourceOwnerAuthentication getResourceOwnerAuthentication() {
    return new ResourceOwnerAuthentication.Builder()
        .withCredentials(new DefaultMuleCredentials(TEST_RESOURCE_OWNER_USERNAME, TEST_RESOURCE_OWNER_PASSWORD.toCharArray()))
        .build();
  }

  protected AccessTokenStoreHolder addAccessTokenToStore(final String accessToken)
      throws OAuth2Exception {
    return addAccessTokenToStore(accessToken, null);
  }

  protected AccessTokenStoreHolder addAccessTokenToStore(final String accessToken,
                                                         final String refreshToken)
      throws OAuth2Exception {
    final Token token =
        new Token.Builder(TEST_CLIENT_ID, accessToken)
            .withRefreshToken(refreshToken)
            .withExpirationInterval(5, SECONDS)
            .build();
    final AuthorizationRequest authorizationRequest = new AuthorizationRequest(TEST_CLIENT_ID,
                                                                               CODE, TEST_REDIRECT_URI,
                                                                               getResourceOwnerAuthentication());
    final AccessTokenStoreHolder accessTokenStoreHolder = new AccessTokenStoreHolder(token,
                                                                                     authorizationRequest, null);
    tokenStore.store(accessTokenStoreHolder);
    return accessTokenStoreHolder;
  }

  protected void updateAccessTokenHolderInOS(AccessTokenStoreHolder accessTokenStoreHolder) {
    tokenStore.store(accessTokenStoreHolder);
  }

  protected MuleContext getMuleContextOfTestedApplication() {
    return AbstractMuleContextTestCase.muleContext;
  }

  protected void assertHasFormFieldContaining(final String htmlBody, final String value) {
    assertThat("form value not found", htmlBody, containsString("value=\"" + value + "\""));
  }

  protected String getAuthorizationEndpointUrl() {
    return buildURL("/authorize");
  }

  protected String getTokenEndpointURL() {
    return buildURL("/token");
  }

  protected String getProtectedResourceURL(final String path) {
    return buildURL(path);
  }

  protected String buildURL(final String path) {
    return getProtocol() + "://localhost:" + port.getNumber() + path;
  }

  protected void executeHttpMethodExpectingStatus(final HttpMethod method, final int expectedStatusCode)
      throws IOException {
    httpClient.executeMethod(method);
    assertThat("Expected another status code for response: " + method.getResponseBodyAsString(),
               method.getStatusCode(), equalTo(expectedStatusCode));
  }

  protected String getHeader(final HttpResponse response, String headerKey) {
    return response.getHeaders(headerKey)[0].getValue();
  }

  protected Map<String, List<String>> validateSuccessfulLoginResponse(final HttpMethod method, final String grantType)
      throws UnsupportedEncodingException, URISyntaxException {
    return validateSuccessfulLoginResponse(method.getResponseHeader("Location").getValue(), grantType);
  }

  protected Map<String, List<String>> validateSuccessfulLoginResponse(final HttpResponse response, final String grantType)
      throws UnsupportedEncodingException, URISyntaxException {
    return validateSuccessfulLoginResponse(getHeader(response, "Location"), grantType);
  }

  protected Map<String, List<String>> validateSuccessfulLoginResponse(final String location,
                                                                      final String grantType)
      throws UnsupportedEncodingException, URISyntaxException {
    assertThat(location, startsWith(TEST_REDIRECT_URI));

    final URI locationUri = new URI(location);
    if (CODE_PARAMETER.equals(grantType)) {
      assertThat("code grant type location has query: " + locationUri, locationUri.getQuery(), notNullValue());
    } else if (ACCESS_TOKEN_PARAMETER.equals(grantType)) {
      assertThat("token grant type location has no query: " + locationUri, locationUri.getQuery(), nullValue());
      assertThat("token grant type location has fragment: " + locationUri, locationUri.getFragment(), notNullValue());
    }

    final Map<String, List<String>> urlParameters = decodeParameters(location);
    final List<String> grant = urlParameters.get(grantType);
    assertThat("Grant type " + grantType + " found in location: " + location, grant,
               both(not(empty())).and(notNullValue()));
    return urlParameters;
  }


  protected Map<String, Object> validateSuccessfulTokenResponseNoScopeNoRefresh(final Map<String, Object> body)
      throws IOException {
    return validateSuccessfulTokenResponseNoRefresh(body, null);
  }

  protected Map<String, Object> validateSuccessfulTokenResponseNoRefresh(final Map<String, Object> body,
                                                                         final String expectedScope)
      throws IOException {
    return validateSuccessfulTokenResponse(body, expectedScope, false);
  }

  protected Map<String, Object> validateSuccessfulTokenResponseNoScope(final Map<String, Object> body,
                                                                       final boolean hasRefreshToken)
      throws IOException {
    return validateSuccessfulTokenResponse(body, null, hasRefreshToken);
  }

  protected Map<String, Object> validateSuccessfulTokenResponse(final Map<String, Object> body,
                                                                final String expectedScope,
                                                                final boolean hasRefreshToken)
      throws IOException {
    @SuppressWarnings("unchecked")
    final Map<String, Object> response = body;


    assertThat(response, hasKey("access_token"));
    assertThat(response, hasKey("token_type"));
    assertThat(response, hasKey("expires_in"));

    if (expectedScope == null) {
      assertThat(response, not(hasKey("scope")));
    } else {
      assertThat(response, hasKey("scope"));
    }

    assertThat(hasRefreshToken, equalTo(response.containsKey("refresh_token")));

    return response;
  }

  protected String getValidBasicAuthHeaderValue(final String username, final String password)
      throws IOException {
    return "Basic " + encodeBytes((URLEncoder.encode(username, UTF_8) + ":" + URLEncoder.encode(password, UTF_8)).getBytes());
  }

  protected PostMethod postOAuthClientRequestExpectingStatus(final OAuthClientRequest request,
                                                             final int expectedStatus)
      throws IOException {
    PostMethod postMethod = getPostOAuthClientRequest(request);
    executeHttpMethodExpectingStatus(postMethod, expectedStatus);
    return postMethod;
  }

  protected PostMethod getPostOAuthClientRequest(final OAuthClientRequest request) throws UnsupportedEncodingException {
    final PostMethod postMethod = new PostMethod(request.getLocationUri());

    postMethod.setRequestEntity(new StringRequestEntity(request.getBody(),
                                                        APPLICATION_X_WWW_FORM_URLENCODED.toRfcString(),
                                                        Charset.defaultCharset().toString()));

    if (MapUtils.isNotEmpty(request.getHeaders())) {
      for (final Entry<String, String> header : request.getHeaders().entrySet()) {
        postMethod.setRequestHeader(header.getKey(), header.getValue());
      }
    }

    return postMethod;
  }

  protected GetMethod getOAuthClientRequestExpectingStatus(final OAuthClientRequest request,
                                                           final int expectedStatus)
      throws IOException {
    final GetMethod getMethod = new GetMethod(request.getLocationUri());
    if (MapUtils.isNotEmpty(request.getHeaders())) {
      for (final Entry<String, String> header : request.getHeaders().entrySet()) {
        getMethod.setRequestHeader(header.getKey(), header.getValue());
      }
    }

    executeHttpMethodExpectingStatus(getMethod, expectedStatus);
    return getMethod;
  }

  protected static Map<String, List<String>> decodeParameters(final String urlOrEncodedParameters)
      throws UnsupportedEncodingException {
    String encodedParams = urlOrEncodedParameters;
    if (StringUtils.contains(urlOrEncodedParameters, "?")) {
      encodedParams = StringUtils.substringAfterLast(encodedParams, "?");
    } else if (StringUtils.contains(urlOrEncodedParameters, "#")) {
      encodedParams = StringUtils.substringAfterLast(encodedParams, "#");
    }

    final Map<String, List<String>> params = new HashMap<String, List<String>>();
    for (final String param : encodedParams.split("&")) {
      final String pair[] = param.split("=");
      final String key = URLDecoder.decode(pair[0], "UTF-8");
      String value = "";
      if (pair.length > 1) {
        value = URLDecoder.decode(pair[1], "UTF-8");
      }
      List<String> values = params.get(key);
      if (values == null) {
        values = new ArrayList<String>();
        params.put(key, values);
      }
      values.add(value);
    }
    return params;
  }

  protected static void assertEqualJsonObj(String expected, HttpResponse response) throws Exception {
    assertThat(response.getHeaders(CONTENT_TYPE)[0].getValue(), equalTo(APPLICATION_JSON));

    compareJsonStrings(expected, IOUtils.toString(response.getEntity().getContent()));
  }

  protected static void assertEqualJsonObj(String expected, PostMethod postMethod) throws Exception {
    assertThat(postMethod.getResponseHeaders(CONTENT_TYPE)[0].getValue(), is(equalTo(APPLICATION_JSON)));

    String actualJsonString = postMethod.getResponseBodyAsString();
    compareJsonStrings(expected, actualJsonString);
  }

  private static void compareJsonStrings(String expected, String actual) {
    JsonParser parser = new JsonParser();
    JsonElement expectedObj = parser.parse(expected);
    JsonElement actualObj = parser.parse(actual);

    assertThat(actualObj, is(equalTo(expectedObj)));
  }

  protected void updateClientInOS() {
    clientStore.addClient(client, false);
  }

  protected void accessProtectedResource(final String accessToken) throws Exception {
    accessProtectedResource(accessToken, PROTECTED_RESOURCE_PATH);
  }

  protected void accessProtectedResource(final String accessToken, final String protectedResourceEndpoint) throws Exception {
    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL(protectedResourceEndpoint)
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, SC_OK);
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE), Matchers.is(nullValue()));
    assertThat(getProtectedResource.getResponseBodyAsString(), Matchers.is(equalTo(PROTECTED_RESOURCE_CONTENT)));
  }
}
