/*
 * (c) 2003-2019 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.error;

import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.CLIENT_CREDENTIALS;
import static org.apache.http.HttpStatus.SC_BAD_REQUEST;
import static org.apache.http.HttpStatus.SC_UNAUTHORIZED;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;

import com.mulesoft.modules.oauth2.provider.AbstractOAuth2ProviderModuleTestCase;

import java.util.Collections;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.http.HttpHeaders;
import org.junit.Before;
import org.junit.Test;

public class OAuth2ProviderClientCredentialsErrorsTestCase extends AbstractOAuth2ProviderModuleTestCase {


  @Override
  protected String doGetConfigFile() {
    return "client-credentials-errors-config.xml";
  }

  @Before
  public void setupClient() {
    client.getAuthorizedGrantTypes().add(CLIENT_CREDENTIALS);
    updateClientInOS();
  }

  private OAuthClientRequest.TokenRequestBuilder baseRequest() {
    return OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setParameter("grant_type", "client_credentials");
  }

  private void assertErrorInRequest(OAuthClientRequest oAuthClientRequest, String expectedError, String expectedDescription,
                                    int expectedErrorCode)
      throws Exception {
    PostMethod response = postOAuthClientRequestExpectingStatus(oAuthClientRequest, expectedErrorCode);
    assertThat(response.getResponseBodyAsString(), containsString("\"error\":\"" + expectedError));
    assertThat(response.getResponseBodyAsString(), containsString("\"error_description\":\"" + expectedDescription));
  }

  private void assertErrorInRequest(OAuthClientRequest oAuthClientRequest, String expectedError, String expectedDescription)
      throws Exception {
    assertErrorInRequest(oAuthClientRequest, expectedError, expectedDescription, SC_BAD_REQUEST);
  }

  private void assertInvalidRequest(OAuthClientRequest oAuthClientRequest, String expectedDescription) throws Exception {
    assertErrorInRequest(oAuthClientRequest, "invalid_request", expectedDescription);
  }

  private void assertInvalidClient(OAuthClientRequest oAuthClientRequest, String expectedDescription) throws Exception {
    assertErrorInRequest(oAuthClientRequest, "invalid_client", expectedDescription);
  }

  @Test
  public void missingAuthentication() throws Exception {
    assertInvalidClient(baseRequest().buildBodyMessage(), "No client identification nor authentication found");
  }

  @Test
  public void multipleAuthenticationMethods() throws Exception {
    OAuthClientRequest request = baseRequest().setClientId(TEST_CLIENT_ID).setClientSecret(TEST_CLIENT_SECRET).buildBodyMessage();
    request.setHeaders(Collections.singletonMap(HttpHeaders.AUTHORIZATION,
                                                getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));
    assertInvalidRequest(request, "Multiple client authentications found");
  }

  @Test
  public void unregisteredClientIdWithSecret() throws Exception {
    OAuthClientRequest request =
        baseRequest().setClientId("unregisteredClient").setClientSecret(TEST_CLIENT_SECRET).buildBodyMessage();
    assertInvalidClient(request, "Invalid credentials");
  }

  @Test
  public void unregisteredClientIdInAuthenticationHeader() throws Exception {
    OAuthClientRequest request = baseRequest().buildBodyMessage();
    request.setHeaders(Collections.singletonMap(HttpHeaders.AUTHORIZATION,
                                                getValidBasicAuthHeaderValue("unregisteredClientId", TEST_CLIENT_PASSWORD)));
    assertErrorInRequest(request, "invalid_client", "Invalid credentials", SC_UNAUTHORIZED);
  }

  @Test
  public void unauthorizedClientWithSecret() throws Exception {
    assertInvalidClient(baseRequest().setClientId(TEST_CLIENT_ID).setClientSecret("wrongSecret").buildBodyMessage(),
                        "Invalid credentials");
  }

  @Test
  public void unauthorizedClientWithAuthorizationHeader() throws Exception {
    OAuthClientRequest request = baseRequest().buildBodyMessage();
    request.setHeaders(Collections.singletonMap(HttpHeaders.AUTHORIZATION,
                                                getValidBasicAuthHeaderValue(TEST_CLIENT_ID, "wrongSecret")));
    assertErrorInRequest(request, "invalid_client", "Invalid credentials", SC_UNAUTHORIZED);
  }

}
