/*
 * (c) 2003-2019 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.error;

import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.AUTHORIZATION_CODE;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import org.mule.runtime.api.event.Event;

import com.mulesoft.modules.oauth2.provider.AbstractOAuth2ProviderModuleTestCase;

import java.util.ArrayList;
import java.util.List;

import io.qameta.allure.Description;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class OAuth2ProviderErrorsTestCase extends AbstractOAuth2ProviderModuleTestCase {

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
  }

  @Override
  protected String doGetConfigFile() {
    return "provider-errors-config.xml";
  }

  @Description("Try to revoke a token that does not exist and see that the correct error is raised")
  @Test
  public void revokeMissingToken() throws Exception {
    Event event = flowRunner("revokeMissingToken").withPayload("MISSING_TOKEN").run();
    assertThat(event.getMessage().getPayload().getValue(), is(equalTo("INVALID_TOKEN")));
  }

  @Description("Validate an empty token and see that it's invalid")
  @Test
  public void validateTokenEmptyToken() throws Exception {
    Event event = flowRunner("validateTokenEmptyToken").run();
    assertThat(event.getMessage().getPayload().getValue(), is(equalTo("UNAUTHORIZED_TOKEN")));
  }

  @Description("Validate a token that was never granted by the provider")
  @Test
  public void validateUnexistentToken() throws Exception {
    Event event = flowRunner("validateUnexistentToken").withVariable("ACCESS_TOKEN", randomAlphanumeric(20)).run();
    assertThat(event.getMessage().getPayload().getValue(), is(equalTo("UNAUTHORIZED_TOKEN")));
  }

  @Description("Validate a token with unmatching scopes with the ones registered and see that if fails")
  @Test
  public void validateUnmatchingScopes() throws Exception {
    String token = randomAlphanumeric(20);
    addAccessTokenToStore(token);
    List<String> scopes = new ArrayList<>();
    Event event = flowRunner("validateUnmatchingScopesToken")
        .withVariable("ACCESS_TOKEN", token)
        .withVariable("SCOPES", scopes).run();
    assertThat(event.getMessage().getPayload().getValue(), is("valid scopes"));
    scopes = singletonList(USER_SCOPE);
    event = flowRunner("validateUnmatchingScopesToken")
        .withVariable("ACCESS_TOKEN", token)
        .withVariable("SCOPES", scopes).run();
    assertThat(event.getMessage().getPayload().getValue(), is(equalTo("UNAUTHORIZED_TOKEN")));
  }

  @Description("Validate token with unmatching resource owner roles and see that it fails with the expected error")
  @Test
  public void validateUnmatchingRoles() throws Exception {
    String token = randomAlphanumeric(20);
    addAccessTokenToStore(token);
    List<String> resourceOwnerRoles = new ArrayList<>();
    Event event = flowRunner("validateUnmatchingRolesToken")
        .withVariable("ACCESS_TOKEN", token)
        .withVariable("ROLES", resourceOwnerRoles)
        .run();
    assertThat(event.getMessage().getPayload().getValue(), is("valid roles"));
    resourceOwnerRoles = singletonList("RESOURCE_OWNER");
    event = flowRunner("validateUnmatchingRolesToken")
        .withVariable("ACCESS_TOKEN", token)
        .withVariable("ROLES", resourceOwnerRoles)
        .run();
    assertThat(event.getMessage().getPayload().getValue(), is(equalTo("UNAUTHORIZED_TOKEN")));

  }

  @Description("Create client with existent clientId and no override flag and see that it fails with the expected error")
  @Test
  public void createExistentClient() throws Exception {
    Event event = flowRunner("createExistentClient").withVariable("CLIENT_ID", TEST_CLIENT_ID).run();
    assertThat(event.getMessage().getPayload().getValue(), is(equalTo("CLIENT_ALREADY_EXISTS")));
  }

  @Description("Create client with authorizationCode or implicit grant type but no redirection uri")
  @Test
  public void createClientNoRedirectUri() throws Exception {
    //Test a grant type that does not exist
    Event event = flowRunner("createClientNoRedirectionUri").withVariable("GRANT_TYPES", singletonList(AUTHORIZATION_CODE)).run();
    assertThat(event.getMessage().getPayload().getValue(), is(equalTo("INVALID_CONFIGURATION")));

  }

  @Description("Create client with invalid grant type and see that it fails with the expected error")
  @Test
  public void createClientWrongGrantType() throws Exception {
    expectedException.expectMessage("RequestGrantType");
    flowRunner("createClientWrongGrantType").run();
  }


  @Description("Create client with no secret whit CONFIDENTIAL type and see that it fails")
  @Test
  public void createConfidentialClientNoSecret() throws Exception {
    Event event = flowRunner("createConfidentialClientNoSecret").run();
    assertThat(event.getMessage().getPayload().getValue(), is(equalTo("INVALID_CONFIGURATION")));
  }

  @Description("Delete not existent client and see that if fails with the expected exception")
  @Test
  public void deleteNotExistentClient() throws Exception {
    Event event = flowRunner("deleteNotExistentClient").run();
    assertThat(event.getMessage().getPayload().getValue(), is(equalTo("NO_SUCH_CLIENT")));
  }

}
