/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;

import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BEARER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.CLIENT_CREDENTIALS;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.PASSWORD;
import static com.mulesoft.modules.oauth2.provider.api.client.ClientType.PUBLIC;
import static net.smartam.leeloo.common.OAuth.OAUTH_CLIENT_ID;
import static net.smartam.leeloo.common.OAuth.OAUTH_CLIENT_SECRET;
import static net.smartam.leeloo.common.OAuth.OAUTH_CODE;
import static net.smartam.leeloo.common.OAuth.OAUTH_GRANT_TYPE;
import static net.smartam.leeloo.common.OAuth.OAUTH_REDIRECT_URI;
import static net.smartam.leeloo.common.message.types.GrantType.AUTHORIZATION_CODE;
import static net.smartam.leeloo.common.message.types.GrantType.NONE;
import static org.apache.http.HttpHeaders.CONTENT_TYPE;
import static org.apache.http.HttpStatus.SC_BAD_REQUEST;
import static org.apache.http.HttpStatus.SC_METHOD_NOT_ALLOWED;
import static org.apache.http.HttpStatus.SC_MOVED_TEMPORARILY;
import static org.apache.http.HttpStatus.SC_OK;
import static org.apache.http.HttpStatus.SC_UNAUTHORIZED;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.both;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.hasEntry;
import static org.hamcrest.Matchers.hasKey;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.isEmptyString;
import static org.hamcrest.Matchers.lessThan;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.AllOf.allOf;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.TOO_MANY_REQUESTS;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.UNAUTHORIZED;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;
import static org.mule.runtime.http.api.HttpHeaders.Names.WWW_AUTHENTICATE;
import static org.mule.runtime.http.api.HttpHeaders.Values.APPLICATION_X_WWW_FORM_URLENCODED;
import static org.mule.runtime.api.metadata.MediaType.HTML;

import org.mule.runtime.api.event.Event;

import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;

import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import net.smartam.leeloo.common.message.types.GrantType;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.PutMethod;
import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Test;

public class OAuth2ProviderModuleCoreTestCase extends AbstractOAuth2ProviderModuleTestCase {

  private static final String CLIENT_RESTRICTED_RESOURCE_PATH = "/client_only";

  @Override
  protected String doGetConfigFile() {
    return "oauth2-core-tests-http-config.xml";
  }

  @Override
  protected String buildURL(final String path) {
    return getProtocol() + "://localhost:" + port.getNumber() + path;
  }

  @Test
  public void accessLoginPageBadMethod() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .buildQueryMessage();

    final PutMethod putAuthorization = new PutMethod(authorizationRequest.getLocationUri());
    executeHttpMethodExpectingStatus(putAuthorization, SC_METHOD_NOT_ALLOWED);
  }

  @Test
  public void accessLoginPageEmptyRequest() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    executeHttpMethodExpectingStatus(getAuthorization, SC_BAD_REQUEST);

    final String responseBody = getAuthorization.getResponseBodyAsString();
    final Map<String, List<String>> urlParameters = decodeParameters(responseBody);
    assertThat(urlParameters.get("error").get(0), is(equalTo("unsupported_response_type")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Missing mandatory parameter: response_type")));
  }

  @Test
  public void accessLoginPageBadResponseType() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("_bad_")
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    executeHttpMethodExpectingStatus(getAuthorization, SC_BAD_REQUEST);

    final String responseBody = getAuthorization.getResponseBodyAsString();
    final Map<String, List<String>> urlParameters = decodeParameters(responseBody);
    assertThat(urlParameters.get("error").get(0), is(equalTo("unsupported_response_type")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Response type '_bad_' is not supported")));
  }

  @Test
  public void accessLoginPageBadClientId() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId("_bad_")
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    executeHttpMethodExpectingStatus(getAuthorization, SC_BAD_REQUEST);

    final String responseBody = getAuthorization.getResponseBodyAsString();
    final Map<String, List<String>> urlParameters = decodeParameters(responseBody);
    assertThat(urlParameters.get("error").get(0), is(equalTo("unauthorized_client")));
  }

  @Test
  public void accessLoginPageBadRedirectUri() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI("_bad_")
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    executeHttpMethodExpectingStatus(getAuthorization, SC_BAD_REQUEST);

    final String responseBody = getAuthorization.getResponseBodyAsString();
    final Map<String, List<String>> urlParameters = decodeParameters(responseBody);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_redirection_uri")));
  }

  @Test
  public void accessLoginPageWithScopeFailureAuthorizationCodeGrant() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setScope(TEST_SCOPE)
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    executeHttpMethodExpectingStatus(getAuthorization, SC_MOVED_TEMPORARILY);

    final String location = getAuthorization.getResponseHeader("Location").getValue();
    assertThat(location, is(not(nullValue())));

    final URI locationUri = new URI(location);
    assertThat("authorization code grant type location has query", locationUri.getQuery(), is(not(nullValue())));

    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_scope")));
  }

  @Test
  public void accessLoginPageWithScopeFailureImplicitGrant() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("token")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setScope(TEST_SCOPE)
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    executeHttpMethodExpectingStatus(getAuthorization, SC_MOVED_TEMPORARILY);

    final String location = getAuthorization.getResponseHeader("Location").getValue();
    assertThat(location, is(not(nullValue())));

    final URI locationUri = new URI(location);
    assertThat("token grant type location has no query", locationUri.getQuery(), is(nullValue()));
    assertThat("token grant type location has fragment", locationUri.getFragment(), is(not(nullValue())));

    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_scope")));
  }

  @Test
  public void accessLoginPageSuccess() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildQueryMessage();
    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    executeHttpMethodExpectingStatus(getAuthorization, SC_OK);

    assertThat(getAuthorization.getResponseHeader(CONTENT_TYPE).getValue(), is(equalTo(HTML.toRfcString())));

    final String responseBody = getAuthorization.getResponseBodyAsString();
    assertThat(responseBody, containsString("<html>"));
    assertHasFormFieldContaining(responseBody, "code");
    assertHasFormFieldContaining(responseBody, TEST_CLIENT_ID);
    assertHasFormFieldContaining(responseBody, TEST_REDIRECT_URI);
  }

  @Test
  public void validateCredentialsNoParamProvided() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .buildBodyMessage();

    final PostMethod postCredentials = postOAuthClientRequestExpectingStatus(authorizationRequest, SC_BAD_REQUEST);

    assertThat(postCredentials.getResponseHeader("Location"), is(nullValue()));
    final String responseBody = postCredentials.getResponseBodyAsString();
    final Map<String, List<String>> urlParameters = decodeParameters(responseBody);
    assertThat(urlParameters.get("error").get(0), is(equalTo("unsupported_response_type")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Missing mandatory parameter: response_type")));
  }

  @Test
  public void validateCredentialsNoCredentialsProvided() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postCredentials = postOAuthClientRequestExpectingStatus(authorizationRequest, SC_MOVED_TEMPORARILY);

    final String location = postCredentials.getResponseHeader("Location").getValue();
    assertThat(location, is(not(nullValue())));
    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_request")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Missing mandatory parameter: username")));
    final String responseBody = postCredentials.getResponseBodyAsString();
    assertThat(responseBody, isEmptyString());
  }

  @Test
  public void validateCredentialsInvalidCredentialsAuthorizationCodeGrant() throws Exception {
    doValidateCredentialsInvalidCredentials("code");
  }

  @Test
  public void validateCredentialsInvalidCredentialsImplicitGrant() throws Exception {
    doValidateCredentialsInvalidCredentials("token");
  }

  @Test
  public void validateCredentialsValidCredentials() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final PostMethod postCredentials = postOAuthClientRequestExpectingStatus(authorizationRequest, SC_MOVED_TEMPORARILY);

    validateSuccessfulLoginResponse(postCredentials, "code");
  }

  @Test
  public void validateCredentialsValidCredentialsWithState() throws Exception {
    final String testState = RandomStringUtils.randomAlphanumeric(10);

    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setState(testState)
        .buildBodyMessage();

    final PostMethod postCredentials = postOAuthClientRequestExpectingStatus(authorizationRequest, SC_MOVED_TEMPORARILY);

    final Map<String, List<String>> urlParameters = validateSuccessfulLoginResponse(postCredentials, "code");
    assertThat(urlParameters.get("state").get(0), is(equalTo(testState)));
  }

  @Test
  public void tokenExchangeEmptyRequest() throws Exception {
    // Ignore this test for the new HTTP module, as it fails because of a bug that was fixed in 3.6.2 (MULE-8338)
    //Assume.assumeFalse(transport.equals("httpn"));

    final PostMethod postToken = new PostMethod(getTokenEndpointURL());
    postToken.setRequestHeader(CONTENT_TYPE, APPLICATION_X_WWW_FORM_URLENCODED.toRfcString());

    executeHttpMethodExpectingStatus(postToken, SC_BAD_REQUEST);

    assertEqualJsonObj(
                       "{\"error\":\"invalid_request\",\"error_description\":\"Missing mandatory parameter: grant_type\"}",
                       postToken);
  }

  @Test
  public void tokenExchangeUnsupportedGrantType() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(NONE)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_BAD_REQUEST);

    assertEqualJsonObj(
                       "{\"error\":\"unsupported_grant_type\",\"error_description\":\"Grant type 'none' is not supported\"}",
                       postToken);
  }

  @Test
  public void tokenExchangeNoCredentials() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_BAD_REQUEST);

    assertEqualJsonObj("{\"error\":\"invalid_client\",\"error_description\":\"Invalid credentials\"}",
                       postToken);
  }

  @Test
  public void tokenExchangeBrokenAuthorization() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .buildBodyMessage();

    oAuthClientRequest.setHeaders(Collections.singletonMap(AUTHORIZATION, "_broken_"));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_BAD_REQUEST);

    assertEqualJsonObj(
                       "{\"error\":\"invalid_request\",\"error_description\":\"Invalid 'Authorization' header\"}",
                       postToken);
  }

  @Test
  public void tokenExchangeBadAuthorization() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .buildBodyMessage();

    oAuthClientRequest.setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue("_bad_", "_bad_")));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_UNAUTHORIZED);

    assertThat(postToken.getResponseBodyAsString(), containsString("\"error\":\"invalid_client\""));
  }

  @Test
  public void tokenExchangeInvalidRequestUri() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setCode("_ignored_")
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_BAD_REQUEST);

    assertEqualJsonObj(
                       "{\"error\":\"invalid_redirection_uri\",\"error_description\":\"Missing mandatory parameter: redirect_uri\"}",
                       postToken);
  }

  @Test
  public void tokenExchangeInvalidGrant() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setCode("_invalid_")
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_BAD_REQUEST);

    assertEqualJsonObj("{\"error\":\"invalid_grant\",\"error_description\":\"Authorization code is invalid or expired\"}",
                       postToken);
  }

  @Test
  public void tokenExchangeMultipleAuthentications() throws Exception {

    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setScope(USER_SCOPE)
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_BAD_REQUEST);

    assertEqualJsonObj(
                       "{\"error\":\"invalid_request\",\"error_description\":\"Multiple client authentications found\"}",
                       postToken);
  }

  @Test
  public void tokenExchangeExpiredAuthorizationCode() throws Exception {
    tokenExchangeValidUsernamePassword(); //Run so that the authentication code is consumed
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_BAD_REQUEST);

    assertEqualJsonObj(
                       "{\"error\":\"invalid_grant\",\"error_description\":\"Authorization code is invalid or expired\"}",
                       postToken);
  }

  @Test
  public void tokenExchangeClientNotFoundInSecurityProvider() throws Exception {
    client.setPrincipal(null);

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    oAuthClientRequest.setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, "_bad_")));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_UNAUTHORIZED);
    assertThat(postToken.getResponseHeader(WWW_AUTHENTICATE), is(not(nullValue())));

    assertEqualJsonObj("{\"error\":\"invalid_client\",\"error_description\":\"Invalid credentials\"}",
                       postToken);
  }

  @Test
  public void tokenExchangeClientIdIsValidSecurityProviderPrincipal() throws Exception {
    setupClient(TEST_CLIENT_OPTIONAL_PRINCIPAL, null);

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
  }

  @Test
  public void tokenExchangeValidUsernamePassword() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
  }

  @Test
  public void tokenExchangeValidBasicAuth() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
  }

  @Test
  public void tokenExchangeValidClientSecret() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
  }

  @Test
  public void tokenExchangePublicClient() throws Exception {
    client.setSecret(null);
    client.setType(PUBLIC);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
  }

  @Test
  public void tokenRequestBadClientAuthorization() throws Exception {
    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.PASSWORD)
        .setUsername(TEST_RESOURCE_OWNER_USERNAME)
        .setPassword(TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    oAuthClientRequest.setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue("_bad_", "_bad_")));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_UNAUTHORIZED);

    assertThat(postToken.getResponseBodyAsString(), containsString("\"error\":\"invalid_client\""));
  }

  @Test
  public void tokenRequestBadResourceOwnerCredentials() throws Exception {
    client.getAuthorizedGrantTypes().add(PASSWORD);
    updateClientInOS();


    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.PASSWORD)
        .setUsername(TEST_RESOURCE_OWNER_USERNAME)
        .setPassword("_bad_")
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_BAD_REQUEST);

    assertThat(postToken.getResponseBodyAsString(), containsString("\"error\":\"access_denied\""));
  }

  @Test
  public void accessProtectedResourceWithoutToken() throws Exception {
    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL(PROTECTED_RESOURCE_PATH));
    executeHttpMethodExpectingStatus(getProtectedResource, UNAUTHORIZED.getStatusCode());
    assertThat(getProtectedResource.getStatusCode(), is(equalTo(UNAUTHORIZED.getStatusCode())));
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE), is(not(nullValue())));
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE).getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }

  @Test
  public void accessProtectedResourceWithBadAccessToken() throws Exception {
    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL(PROTECTED_RESOURCE_PATH)
        + "?access_token=_bad_");
    executeHttpMethodExpectingStatus(getProtectedResource, SC_UNAUTHORIZED);

    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE), is(not(nullValue())));
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE).getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }

  @Test
  public void accessProtectedResourceWithExpiredAccessToken() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);

    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL(PROTECTED_RESOURCE_PATH)
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, SC_UNAUTHORIZED);

    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE), is(not(nullValue())));
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE).getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }

  @Test
  public void accessProtectedResourceWithAccessTokenQueryParam() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);
    addAccessTokenToStore(accessToken);

    accessProtectedResource(accessToken);
  }

  @Test
  public void accessProtectedResourceWithBearerHeader() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);
    addAccessTokenToStore(accessToken);

    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL(PROTECTED_RESOURCE_PATH + "-with-bearer"));
    getProtectedResource.addRequestHeader("Authorization", "Bearer " + accessToken);

    executeHttpMethodExpectingStatus(getProtectedResource, SC_OK);
    assertThat(getProtectedResource.getResponseBodyAsString(), is(equalTo(PROTECTED_RESOURCE_CONTENT)));
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE), is(nullValue()));
  }

  @Test
  public void performAuthorizationCodeOAuth2DanceAndAccessProtectedResource() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final PostMethod postCredentials = postOAuthClientRequestExpectingStatus(authorizationRequest, SC_MOVED_TEMPORARILY);

    final Map<String, List<String>> authorizationResponse = validateSuccessfulLoginResponse(postCredentials, "code");
    final String authorizationCode = authorizationResponse.get("code").get(0);

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(authorizationCode)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
    final String accessToken = (String) tokenResponse.get("access_token");

    accessProtectedResource(accessToken);
  }

  @Test
  public void performImplicitGrantOAuth2DanceAndAccessProtectedResource() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("token")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(authorizationRequest, SC_MOVED_TEMPORARILY);

    final Map<String, List<String>> tokenResponse = validateSuccessfulLoginResponse(postToken, "access_token");
    final String accessToken = tokenResponse.get("access_token").get(0);

    accessProtectedResource(accessToken);
  }

  @Test
  public void performResourceOwnerPasswordCredentialsGrantOAuth2DanceAndAccessProtectedResource()
      throws Exception {
    client.getAuthorizedGrantTypes().add(PASSWORD);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.PASSWORD)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
    final String accessToken = (String) tokenResponse.get("access_token");

    accessProtectedResource(accessToken);
  }

  @Test
  public void performClientCredentialsGrantOAuth2DanceAndAccessProtectedResourceWithClientPassword() throws Exception {
    client.getAuthorizedGrantTypes().add(CLIENT_CREDENTIALS);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setParameter("grant_type", "client_credentials")
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
    final String accessToken = (String) tokenResponse.get("access_token");

    accessProtectedResource(accessToken);
  }

  @Test
  public void performClientCredentialsGrantOAuth2DanceAndAccessProtectedResourceWithClientSecret() throws Exception {
    client.getAuthorizedGrantTypes().add(CLIENT_CREDENTIALS);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setParameter("grant_type", "client_credentials")
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_SECRET)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
    final String accessToken = (String) tokenResponse.get("access_token");

    accessProtectedResource(accessToken);
  }

  @Test
  public void multipleInvalidCredentialsAreRejected() throws Exception {
    final String responseType = "code";

    doValidateCredentialsInvalidCredentials(responseType);
    doValidateCredentialsInvalidCredentials(responseType);

    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType(responseType)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", "__BAD__")
        .buildBodyMessage();

    postOAuthClientRequestExpectingStatus(authorizationRequest, TOO_MANY_REQUESTS.getStatusCode());
  }

  @Test
  public void getAccessTokenWithGETAndFail() throws Exception {

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setParameter(OAUTH_GRANT_TYPE, AUTHORIZATION_CODE.toString())
        .setParameter(OAUTH_CODE, "useless_authentication_code")
        .setParameter(OAUTH_CLIENT_ID, TEST_CLIENT_ID)
        .setParameter(OAUTH_CLIENT_SECRET, TEST_CLIENT_SECRET)
        .setParameter(OAUTH_REDIRECT_URI, TEST_REDIRECT_URI)
        .buildQueryMessage();

    getOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_METHOD_NOT_ALLOWED);


  }

  @Test
  public void tokenAttributesHaveCorrectValues() throws Exception {
    //Get an access token
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final PostMethod postCredentials = postOAuthClientRequestExpectingStatus(authorizationRequest, SC_MOVED_TEMPORARILY);

    final Map<String, List<String>> authorizationResponse = validateSuccessfulLoginResponse(postCredentials, "code");
    final String authorizationCode = authorizationResponse.get("code").get(0);

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(authorizationCode)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest,
                                                                       SC_OK);

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
    final String accessToken = (String) tokenResponse.get("access_token");

    Event event = flowRunner("testAttributes").withVariable("ACCESS_TOKEN", accessToken).run();
    Long tokenTtl = (Long) event.getVariables().get("tokenTtl").getValue();
    assertThat(tokenTtl, is(both(greaterThan(5000000000L)).and(lessThan(10000000000L))));

    event = flowRunner("testAttributes").withVariable("ACCESS_TOKEN", accessToken).run();
    Long tokenTtl2 = (Long) event.getVariables().get("tokenTtl").getValue();
    assertThat(tokenTtl2, is(lessThan(tokenTtl)));

  }

  @Test
  public void payloadIsCorrectlyGenerated() throws Exception {
    //Get an access token
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final PostMethod postCredentials = postOAuthClientRequestExpectingStatus(authorizationRequest, SC_MOVED_TEMPORARILY);

    final Map<String, List<String>> authorizationResponse = validateSuccessfulLoginResponse(postCredentials, "code");
    final String authorizationCode = authorizationResponse.get("code").get(0);

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(AUTHORIZATION_CODE)
        .setCode(authorizationCode)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, SC_OK);

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
    final String accessToken = (String) tokenResponse.get("access_token");

    Event event = flowRunner("validateToken").withVariable("ACCESS_TOKEN", accessToken).run();
    Map<String, String> payload = new Gson().fromJson(event.getMessage().getPayload().getValue().toString(),
                                                      new TypeToken<Map<String, String>>() {}.getType());
    assertThat(payload, allOf(hasKey("expires_in"),
                              hasEntry("client_id", TEST_CLIENT_ID),
                              hasEntry("username", TEST_RESOURCE_OWNER_USERNAME),
                              hasEntry("scope", "")));

  }

  private void doValidateCredentialsInvalidCredentials(final String responseType) throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType(responseType)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", "__BAD__")
        .buildBodyMessage();

    final PostMethod postCredentials = postOAuthClientRequestExpectingStatus(authorizationRequest, SC_MOVED_TEMPORARILY);

    final String location = postCredentials.getResponseHeader("Location").getValue();
    assertThat(location, is(not(nullValue())));
    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("access_denied")));
    final String responseBody = postCredentials.getResponseBodyAsString();
    assertThat(responseBody, isEmptyString());
  }
}
