/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;

import static java.util.Collections.singletonList;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.MOVED_TEMPORARILY;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;

import org.mule.runtime.core.api.MuleContext;
import org.mule.tck.junit4.AbstractMuleContextTestCase;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import net.smartam.leeloo.common.message.types.GrantType;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.junit.Test;

public class OAuth2ProviderModuleCreateClientTestCase extends AbstractOAuth2ProviderModuleTestCase {

  private static final String PROTECTED_RESOURCE_PATH = "/protected";

  @Override
  protected String doGetConfigFile() {
    return "oauth2-create-client-tests-http-config.xml";
  }

  @Override
  protected MuleContext getMuleContextOfTestedApplication() {
    return AbstractMuleContextTestCase.muleContext;
  }

  @Override
  protected String buildURL(final String path) {
    return getProtocol() + "://localhost:" + port.getNumber() + path;
  }

  @Test
  public void performAuthorizationCodeOAuth2DanceAndAccessProtectedResource() throws Exception {
    registerClient("registerClient", "CONFIDENTIAL", TEST_CLIENT_SECRET);
    String accessToken = getAccessToken();
    accessProtectedResource(accessToken);
    deleteClient();
  }

  @Test
  public void registerPublicClientWithNoSecurityProvider() throws Exception {
    registerClient("registerClientNoSecurityProvider", "PUBLIC", null);
  }

  @Test
  public void registerConfidentialClientWithSecretNoSecurityProvider() throws Exception {
    registerClient("registerClientNoSecurityProvider", "CONFIDENTIAL", TEST_CLIENT_SECRET);
  }

  private String getAccessToken() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final PostMethod postCredentials =
        postOAuthClientRequestExpectingStatus(authorizationRequest, MOVED_TEMPORARILY.getStatusCode());

    final Map<String, List<String>> authorizationResponse = validateSuccessfulLoginResponse(
                                                                                            postCredentials, "code");
    final String authorizationCode = authorizationResponse.get("code").get(0);

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.AUTHORIZATION_CODE)
        .setCode(authorizationCode)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, OK.getStatusCode());

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
    final String accessToken = (String) tokenResponse.get("access_token");

    return accessToken;
  }

  private void deleteClient() throws Exception {
    flowRunner("deleteClient").withVariable("CLIENT_ID", TEST_CLIENT_ID).run();
  }

  private void registerClient(String flow, String clientType, String clientSecret) throws Exception {
    flowRunner(flow)
        .withVariable("CLIENT_ID", TEST_CLIENT_ID)
        .withVariable("CLIENT_SECRET", clientSecret)
        .withVariable("CLIENT_PRINCIPAL", TEST_CLIENT_OPTIONAL_PRINCIPAL)
        .withVariable("CLIENT_TYPE", clientType)
        .withVariable("AUTHORIZED_GRANT_TYPES", singletonList("AUTHORIZATION_CODE"))
        .withVariable("REDIRECT_URIS", singletonList(TEST_REDIRECT_URI))
        .withPayload(client).run();
  }
}
