/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;


import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BEARER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.AUTHORIZATION_CODE;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.CLIENT_CREDENTIALS;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.PASSWORD;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.AllOf.allOf;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.FORBIDDEN;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.MOVED_TEMPORARILY;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.WWW_AUTHENTICATE;
import static org.mule.runtime.api.metadata.MediaType.HTML;

import com.mulesoft.modules.oauth2.provider.api.token.AccessTokenStoreHolder;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import net.smartam.leeloo.common.message.types.GrantType;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Test;

public class OAuth2ProviderModuleScopeTestCase extends AbstractOAuth2ProviderModuleTestCase {


  @Override
  protected String doGetConfigFile() {
    return "oauth2-scope-http-config.xml";
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    client.getScopes().add(USER_SCOPE);
    client.getAuthorizedGrantTypes().add(AUTHORIZATION_CODE);
    updateClientInOS();
  }

  @Test
  public void accessLoginPageClientAuthorizeNoScope() throws Exception {
    client.getScopes().clear();
    updateClientInOS();

    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    executeHttpMethodExpectingStatus(getAuthorization, MOVED_TEMPORARILY.getStatusCode());

    final String location = getAuthorization.getResponseHeader("Location").getValue();
    assertThat(location, is(not(nullValue())));
    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("temporarily_unavailable")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Client configuration error")));
  }

  @Test
  public void accessLoginPageWithoutScopeFailure() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    executeHttpMethodExpectingStatus(getAuthorization, MOVED_TEMPORARILY.getStatusCode());

    final String location = getAuthorization.getResponseHeader("Location").getValue();
    assertThat(location, is(not(nullValue())));
    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_request")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Missing mandatory parameter: scope")));
  }

  @Test
  public void accessLoginPageWithUnsupportedScopeFailure() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setScope("GUEST")
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    getAuthorization.setFollowRedirects(false);
    executeHttpMethodExpectingStatus(getAuthorization, MOVED_TEMPORARILY.getStatusCode());

    final String location = getAuthorization.getResponseHeader("Location").getValue();
    assertThat(location, is(not(nullValue())));
    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_scope")));
    assertThat(urlParameters.get("error_description"), is(nullValue()));
  }

  @Test
  public void accessLoginPageWithScopeSuccess() throws Exception {
    final String clientName = RandomStringUtils.randomAlphanumeric(20);
    client.setClientName(clientName);
    updateClientInOS();

    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setScope(USER_SCOPE)
        .buildQueryMessage();

    final GetMethod getAuthorization = new GetMethod(authorizationRequest.getLocationUri());
    executeHttpMethodExpectingStatus(getAuthorization, OK.getStatusCode());

    assertThat(getAuthorization.getResponseHeader(CONTENT_TYPE).getValue(), is(equalTo(HTML.toRfcString())));

    final String responseBody = getAuthorization.getResponseBodyAsString();
    assertThat(responseBody, allOf(containsString("<html>"),
                                   containsString("<li>" + USER_SCOPE + "</li>"),
                                   containsString(clientName),
                                   containsString("TestScopedProvider")));
    assertHasFormFieldContaining(responseBody, "code");
    assertHasFormFieldContaining(responseBody, TEST_CLIENT_ID);
    assertHasFormFieldContaining(responseBody, TEST_REDIRECT_URI);
    assertHasFormFieldContaining(responseBody, USER_SCOPE);
  }

  @Test
  public void validateCredentialsMissingScope() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setParameter("client_secret", TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    final PostMethod postCredentials =
        postOAuthClientRequestExpectingStatus(authorizationRequest, MOVED_TEMPORARILY.getStatusCode());

    final String location = postCredentials.getResponseHeader("Location").getValue();
    assertThat(location, is(not(nullValue())));
    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_request")));
    assertThat(urlParameters.get("error_description").get(0), is(equalTo("Missing mandatory parameter: scope")));
  }

  @Test
  public void validateCredentialsUnsupportedScope() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setScope("GUEST")
        .buildBodyMessage();

    final PostMethod postCredentials =
        postOAuthClientRequestExpectingStatus(authorizationRequest, MOVED_TEMPORARILY.getStatusCode());

    final String location = postCredentials.getResponseHeader("Location").getValue();
    assertThat(location, is(not(nullValue())));
    final Map<String, List<String>> urlParameters = decodeParameters(location);
    assertThat(urlParameters.get("error").get(0), is(equalTo("invalid_scope")));
    assertThat(urlParameters.get("error_description"), is(nullValue()));
  }

  @Test
  public void validateCredentialsSuccess() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setScope(USER_SCOPE)
        .buildBodyMessage();

    final PostMethod postCredentials =
        postOAuthClientRequestExpectingStatus(authorizationRequest, MOVED_TEMPORARILY.getStatusCode());

    validateSuccessfulLoginResponse(postCredentials, "code");
  }

  @Test
  public void tokenExchangeSuccess() throws Exception {
    authorizationCodeStoreHolder.getAuthorizationRequest().getScopes().add(USER_SCOPE);
    updateAuthorizationCodeInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.AUTHORIZATION_CODE)
        .setCode(TEST_AUTHORIZATION_CODE)
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, OK.getStatusCode());

    validateSuccessfulTokenResponseNoRefresh(getContentAsMap(postToken), USER_SCOPE);
  }

  @Test
  public void accessScopelessProtectedResourceSuccess() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);
    addAccessTokenToStore(accessToken);

    accessScopelessProtectedResource(accessToken);
  }

  @Test
  public void accessScopedProtectedResourceWithBadScope() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);
    addAccessTokenToStore(accessToken);

    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL("/protected/scopes")
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, FORBIDDEN.getStatusCode());
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE), is(not(nullValue())));
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE).getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }

  @Test
  public void accessScopedProtectedResourceSuccess() throws Exception {
    final String accessToken = RandomStringUtils.randomAlphanumeric(20);
    final AccessTokenStoreHolder accessTokenStoreHolder = addAccessTokenToStore(accessToken);
    accessTokenStoreHolder.getAccessToken().getScopes().add(USER_SCOPE);
    updateAccessTokenHolderInOS(accessTokenStoreHolder);

    accessScopedProtectedResource(accessToken);
  }

  @Test
  public void performAuthorizationCodeOAuth2DanceAndAccessProtectedResources() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("code")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setScope(USER_SCOPE)
        .buildBodyMessage();

    final PostMethod postCredentials =
        postOAuthClientRequestExpectingStatus(authorizationRequest, MOVED_TEMPORARILY.getStatusCode());

    final Map<String, List<String>> authorizationResponse = validateSuccessfulLoginResponse(
                                                                                            postCredentials, "code");
    final String authorizationCode = authorizationResponse.get("code").get(0);

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.AUTHORIZATION_CODE)
        .setCode(authorizationCode)
        .setClientId(TEST_CLIENT_ID)
        .setClientSecret(TEST_CLIENT_SECRET)
        .setRedirectURI(TEST_REDIRECT_URI)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, OK.getStatusCode());

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoRefresh(getContentAsMap(postToken),
                                                                                       USER_SCOPE);
    final String accessToken = (String) tokenResponse.get("access_token");

    accessScopelessProtectedResource(accessToken);
    accessScopedProtectedResource(accessToken);
    accessForbiddenScopedProtectedResource(accessToken);
  }

  @Test
  public void performImplicitGrantOAuth2DanceAndAccessProtectedResource() throws Exception {
    final OAuthClientRequest authorizationRequest = OAuthClientRequest.authorizationLocation(
                                                                                             getAuthorizationEndpointUrl())
        .setResponseType("token")
        .setClientId(TEST_CLIENT_ID)
        .setRedirectURI(TEST_REDIRECT_URI)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setScope(USER_SCOPE)
        .buildBodyMessage();

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(authorizationRequest, MOVED_TEMPORARILY.getStatusCode());

    final Map<String, List<String>> tokenResponse = validateSuccessfulLoginResponse(postToken,
                                                                                    "access_token");
    final String accessToken = tokenResponse.get("access_token").get(0);

    accessScopelessProtectedResource(accessToken);
    accessScopedProtectedResource(accessToken);
    accessForbiddenScopedProtectedResource(accessToken);
  }

  @Test
  public void performResourceOwnerPasswordCredentialsGrantOAuth2DanceAndAccessProtectedResource()
      throws Exception {
    client.getAuthorizedGrantTypes().add(PASSWORD);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.PASSWORD)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .setScope(USER_SCOPE)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, OK.getStatusCode());

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoRefresh(getContentAsMap(postToken),
                                                                                       USER_SCOPE);
    final String accessToken = (String) tokenResponse.get("access_token");

    accessScopelessProtectedResource(accessToken);
    accessScopedProtectedResource(accessToken);
    accessForbiddenScopedProtectedResource(accessToken);
  }

  @Test
  public void performClientCredentialsGrantOAuth2DanceAndAccessProtectedResource() throws Exception {
    client.getAuthorizedGrantTypes().add(CLIENT_CREDENTIALS);
    updateClientInOS();

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setParameter("grant_type", "client_credentials")
        .setScope(USER_SCOPE)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, OK.getStatusCode());

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoRefresh(getContentAsMap(postToken),
                                                                                       USER_SCOPE);
    final String accessToken = (String) tokenResponse.get("access_token");

    accessScopelessProtectedResource(accessToken);
    accessScopedProtectedResource(accessToken);
    accessForbiddenScopedProtectedResource(accessToken);
  }

  private void accessScopelessProtectedResource(final String accessToken) throws IOException {
    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL("/protected/no-scope")
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, OK.getStatusCode());
    assertThat(getProtectedResource.getResponseBodyAsString(), is(equalTo(PROTECTED_RESOURCE_CONTENT)));
  }

  private void accessScopedProtectedResource(final String accessToken) throws IOException {
    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL("/protected/scopes")
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, OK.getStatusCode());
    assertThat(getProtectedResource.getResponseBodyAsString(), is(equalTo(PROTECTED_RESOURCE_CONTENT)));
  }

  private void accessForbiddenScopedProtectedResource(final String accessToken) throws IOException {
    final GetMethod getProtectedResource = new GetMethod(
                                                         getProtectedResourceURL("/protected/forbidden-scope") + "?access_token="
                                                             + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, FORBIDDEN.getStatusCode());
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE), is(not(nullValue())));
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE).getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }
}
