/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;


import static com.mulesoft.modules.oauth2.provider.api.Constants.ProviderGrantType.AUTHORIZATION_CODE;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ProviderGrantType.IMPLICIT;
import static com.mulesoft.modules.oauth2.provider.internal.Utils.tokenize;
import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static org.hamcrest.CoreMatchers.anyOf;
import static org.hamcrest.CoreMatchers.hasItems;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mule.runtime.core.api.util.Base64.encodeBytes;

import com.mulesoft.modules.oauth2.provider.internal.Utils;
import com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException;

import java.util.HashSet;

import org.hamcrest.CoreMatchers;
import org.junit.Test;

public class UtilsTestCase {

  @Test
  public void generateUniqueId() {
    assertThat(Utils.generateUniqueId("salty"), is(CoreMatchers.not(CoreMatchers.nullValue())));
  }

  @Test
  public void extractCredentialsFromAuthorizationHeader() throws Exception {
    assertThat(Utils.extractCredentialsFromAuthorizationHeader(null, "Bar", "utf-8"),
               is(CoreMatchers.nullValue()));
    assertThat(Utils.extractCredentialsFromAuthorizationHeader("", "Bar", "utf-8"),
               is(CoreMatchers.nullValue()));
    assertThat(Utils.extractCredentialsFromAuthorizationHeader("Foo", "Bar", "utf-8"),
               is(CoreMatchers.nullValue()));
    assertThat(Utils.extractCredentialsFromAuthorizationHeader("Foo Ignored", "Bar", "utf-8"),
               is(CoreMatchers.nullValue()));

    final String credentials = "credentials";

    assertThat(Utils.extractCredentialsFromAuthorizationHeader(
                                                               "Basic " + encodeBytes(credentials.getBytes()), "Basic",
                                                               "utf-8"),
               is(credentials));
    assertThat(
               Utils.extractCredentialsFromAuthorizationHeader("Bearer " + credentials, "Bearer", "utf-8"),
               is(credentials));
  }

  @Test
  public void tokenizeScope() {
    assertThat(tokenize(null), is(emptySet()));
    assertThat(tokenize(""), is(emptySet()));
    assertThat(tokenize(" SCOPE1 "), hasItems("SCOPE1"));
    assertThat(tokenize(" SCOPE1  SCOPE2  "), hasItems("SCOPE1", "SCOPE2"));
    assertThat(tokenize(" SCOPE1,SCOPE2  ", ","), hasItems("SCOPE1", "SCOPE2"));
    assertThat(tokenize(" SCOPE1, SCOPE2  ", ","), hasItems("SCOPE1", "SCOPE2"));
  }

  @Test
  public void stringifyScopes() {
    assertThat(Utils.stringifyScopes(null), is(""));
    assertThat(Utils.stringifyScopes(emptySet()), is(""));
    assertThat(Utils.stringifyScopes(new HashSet<>(asList("SCOPE1"))), is("SCOPE1"));
    assertThat(Utils.stringifyScopes(new HashSet<>(asList("SCOPE1", "SCOPE2"))), anyOf(is("SCOPE1 SCOPE2"), is("SCOPE2 SCOPE1")));
  }

  @Test(expected = RequestProcessingException.class)
  public void computeEffectiveScopeMissingRequestedScope() throws RequestProcessingException {
    Utils.computeEffectiveScopeOrFail(emptySet(), tokenize("SCOPE1"), tokenize("SCOPE1 SCOPE2"));
  }

  @Test(expected = RequestProcessingException.class)
  public void computeEffectiveScopeUnsupportedRequestedScope() throws RequestProcessingException {
    Utils.computeEffectiveScopeOrFail(tokenize("SCOPE3"), tokenize("SCOPE1"),
                                      tokenize("SCOPE1 SCOPE2"));
  }

  @Test(expected = RequestProcessingException.class)
  public void computeEffectiveScopeObsoleteClientScopes() throws RequestProcessingException {
    Utils.computeEffectiveScopeOrFail(tokenize("SCOPE1"), tokenize("SCOPE3"),
                                      tokenize("SCOPE1 SCOPE2"));
  }

  @Test
  public void computeEffectiveScope() throws RequestProcessingException {
    assertThat(Utils.computeEffectiveScopeOrFail(tokenize("SCOPE2"),
                                                 tokenize("SCOPE2 SCOPE3"), tokenize("SCOPE1 SCOPE2")),
               is(new HashSet<>(singletonList("SCOPE2"))));
  }

  @Test
  public void urlDecode() {
    assertThat(Utils.urlDecode("a%20b"), is("a b"));
  }

  @Test(expected = IllegalArgumentException.class)
  public void parseProviderGrantTypesFailure() {
    Utils.parseProviderGrantTypes("AUTHORIZATION_CODE ABC");
  }

  @Test
  public void parseProviderGrantTypes() {
    assertThat(Utils.parseProviderGrantTypes(null), is(emptySet()));

    assertThat(Utils.parseProviderGrantTypes("  "), is(emptySet()));

    assertThat(Utils.parseProviderGrantTypes("AUTHORIZATION_CODE, IMPLICIT"),
               is(new HashSet<>(asList(AUTHORIZATION_CODE, IMPLICIT))));
  }
}
