/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.api;

import org.mule.runtime.api.security.Authentication;
import org.mule.runtime.core.api.security.DefaultMuleCredentials;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Authentication object for ResourceOwners
 *
 * @since 1.0.0
 */
public class ResourceOwnerAuthentication implements Authentication {

  private static final long serialVersionUID = 7726405183870224981L;
  private String username;
  private Object principal;
  private Object credentials;
  private Set<String> roles;
  private Map<String, Object> properties;

  private ResourceOwnerAuthentication(Builder builder) {
    this.username = builder.username;
    this.principal = builder.principal;
    this.credentials = builder.credentials;
    this.roles = builder.roles;
    this.properties = builder.properties;
  }

  @Override
  public Object getCredentials() {
    return credentials;
  }

  @Override
  public Object getPrincipal() {
    return principal;
  }

  @Override
  public Map<String, Object> getProperties() {
    return properties;
  }

  @Override
  public Authentication setProperties(Map<String, Object> properties) {
    return new Builder().withUsername(this.username).withPrincipal(this.principal).withCredentials(this.credentials)
        .withRoles(this.roles).withProperties(properties).build();
  }

  public String getUsername() {
    return username;
  }

  public Set<String> getRoles() {
    return roles;
  }

  public static class Builder {

    private String username;
    private Object principal;
    private Object credentials;
    private Set<String> roles = new HashSet<>();
    private Map<String, Object> properties = new HashMap<>();

    public Builder withUsername(String username) {
      this.username = username;
      if (principal == null) {
        principal = username;
      }
      return this;
    }

    public Builder withPrincipal(Object principal) {
      if (principal instanceof String) {
        return withPrincipal((String) principal);
      }
      this.principal = principal;
      return this;
    }

    private Builder withPrincipal(String principal) {
      this.principal = principal;
      if (username == null) {
        this.username = principal;
      }
      return this;
    }

    public Builder withCredentials(Object credentials) {
      if (credentials instanceof DefaultMuleCredentials) {
        return withCredentials((DefaultMuleCredentials) credentials);
      }
      this.credentials = credentials;
      return this;
    }

    private Builder withCredentials(DefaultMuleCredentials credentials) {
      this.credentials = credentials.getPassword();
      if (username == null) {
        withUsername(credentials.getUsername());
      }
      return this;
    }

    public Builder withRoles(Set<String> roles) {
      this.roles = roles;
      return this;
    }

    public Builder withProperties(Map<String, Object> properties) {
      this.properties = properties;
      return this;
    }

    public ResourceOwnerAuthentication build() {
      return new ResourceOwnerAuthentication(this);
    }

  }

}
