/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.api.token;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BEARER;
import static java.lang.System.currentTimeMillis;
import static java.time.Duration.ZERO;
import static java.time.Duration.ofNanos;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.core.api.util.StringUtils.isEmpty;

import java.io.Serializable;
import java.time.Duration;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;


/**
 * Represents an Oauth 2.0 Access Token.
 *
 * @since 1.0.0
 */
public class Token implements Serializable {

  private static final long serialVersionUID = -2295676325723230622L;

  private final String type;
  private final String clientId;
  private final String accessToken;
  private final String refreshToken;
  private final Set<String> scopes;
  private final long expirationInterval;
  private final long timeOfCreation;

  private static final Logger logger = LoggerFactory.getLogger(Token.class);

  private Token(Builder builder) {
    checkArgument(builder != null, "builder can't be null");
    checkArgument(!isEmpty(builder.type), "token type should be set in builder");
    checkArgument(!isEmpty(builder.clientId), "token's clientId should be set in builder");
    checkArgument(!isEmpty(builder.accessToken), "accessToken should be set in builder");
    checkArgument(builder.expirationInterval > 0, "token expiration interval should be bigger than 0");

    type = builder.type;
    clientId = builder.clientId;
    accessToken = builder.accessToken;
    refreshToken = builder.refreshToken;
    scopes = builder.scopes == null ? new HashSet<>() : builder.scopes;
    expirationInterval = builder.expirationInterval;
    timeOfCreation = TimeUnit.MILLISECONDS.toNanos(currentTimeMillis());
    if (logger.isDebugEnabled()) {
      logger.debug("Time of creation: {}, Expiration interval: {}", timeOfCreation, expirationInterval);
    }
  }

  public static class Builder {


    private final String clientId;
    private final String accessToken;
    private String type;
    private String refreshToken;
    private Set<String> scopes;
    private long expirationInterval;

    public Builder(final String accessToken, final Token previousToken) {
      this.clientId = previousToken.getClientId();
      this.accessToken = accessToken;
      this.type = previousToken.getType();
      this.scopes = previousToken.getScopes();
      this.expirationInterval = previousToken.expirationInterval;
    }

    public Builder(final String clientId, final String accessToken) {
      this.clientId = clientId;
      this.accessToken = accessToken;
    }

    public Builder withRefreshToken(String refreshToken) {
      this.refreshToken = refreshToken;
      return this;
    }

    public Builder withScopes(Set<String> scopes) {
      this.scopes = scopes == null ? new HashSet<>() : scopes;
      return this;
    }

    public Builder withExpirationInterval(long intervalTime, TimeUnit intervalTimeUnit) {
      this.expirationInterval = intervalTimeUnit.toNanos(intervalTime);
      return this;
    }

    public Token build() {
      if (type == null) {
        type = HTTP_AUTHORIZATION_SCHEME_BEARER;
      }

      return new Token(this);
    }

  }

  public boolean hasRefreshToken() {
    return !isEmpty(refreshToken);
  }

  public String getClientId() {
    return clientId;
  }

  public String getAccessToken() {
    return accessToken;
  }

  public String getRefreshToken() {
    return refreshToken;
  }

  public Set<String> getScopes() {
    return scopes;
  }

  public String getType() {
    return type;
  }

  public Duration getExpiresIn() {
    long now = TimeUnit.MILLISECONDS.toNanos(currentTimeMillis());
    long elapsedTime = now - timeOfCreation;
    long expiresIn = expirationInterval - elapsedTime;
    if (logger.isDebugEnabled()) {
      logger.debug("Time of creation: {}, Now: {}, Elapsed time: {}, Expiration interval: {}, Expires in: {}", timeOfCreation,
                   now, elapsedTime, expirationInterval, expiresIn);
    }
    return expiresIn <= 0 ? ZERO : ofNanos(expiresIn);
  }



}
