/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal;

import static com.mulesoft.modules.oauth2.provider.api.client.ClientType.CONFIDENTIAL;
import static com.mulesoft.modules.oauth2.provider.internal.OAuth2ProviderConfiguration.WWW_AUTHENTICATE_HEADER_VALUE;
import static com.mulesoft.modules.oauth2.provider.internal.Utils.stringifyScopes;
import static com.mulesoft.modules.oauth2.provider.internal.error.OAuth2ProviderError.OAUTH_SERVER_SECURITY;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.api.meta.ExpressionSupport.REQUIRED;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.core.api.util.StringUtils.isBlank;
import static org.mule.runtime.core.api.util.StringUtils.isEmpty;
import static org.mule.runtime.extension.api.annotation.param.MediaType.APPLICATION_JSON;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.FORBIDDEN;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.UNAUTHORIZED;
import static org.mule.runtime.http.api.HttpHeaders.Names.WWW_AUTHENTICATE;
import org.mule.extension.http.api.HttpListenerResponseAttributes;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.security.SecurityProviderNotFoundException;
import org.mule.runtime.api.security.SecurityException;
import org.mule.runtime.api.security.UnknownAuthenticationTypeException;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.security.AuthenticationHandler;

import com.google.gson.JsonObject;
import com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType;
import com.mulesoft.modules.oauth2.provider.api.client.Client;
import com.mulesoft.modules.oauth2.provider.api.client.ClientAlreadyExistsException;
import com.mulesoft.modules.oauth2.provider.api.client.ClientType;
import com.mulesoft.modules.oauth2.provider.api.token.AccessTokenStoreHolder;
import com.mulesoft.modules.oauth2.provider.api.token.Token;
import com.mulesoft.modules.oauth2.provider.api.token.TokenStore;
import com.mulesoft.modules.oauth2.provider.api.client.NoSuchClientException;
import com.mulesoft.modules.oauth2.provider.api.token.UnauthorizedTokenException;
import com.mulesoft.modules.oauth2.provider.internal.error.CreateClientErrorProvider;
import com.mulesoft.modules.oauth2.provider.internal.error.DeleteClientErrorProvider;
import com.mulesoft.modules.oauth2.provider.api.exception.OAuth2ConfigurationException;
import com.mulesoft.modules.oauth2.provider.internal.error.RevokeTokenErrorProvider;
import com.mulesoft.modules.oauth2.provider.internal.error.ValidateTokenErrorProvider;
import com.mulesoft.modules.oauth2.provider.api.token.InvalidTokenException;
import com.mulesoft.modules.oauth2.provider.api.ResourceOwnerAuthentication;
import com.mulesoft.modules.oauth2.provider.internal.token.ForbiddenSecurityException;
import com.mulesoft.modules.oauth2.provider.internal.token.TokenAuthentication;
import com.mulesoft.modules.oauth2.provider.internal.token.UnauthorizedSecurityException;

import java.util.Set;

public class OAuth2ProviderOperations {


  /**
   * <p>
   * Checks that a valid access token is provided.
   * </p>
   * <p/>
   * {@sample.xml ../../../doc/oauth2-provider-connector.xml.sample oauth2-provider:validate}
   *
   * @param oAuth2ProviderConfiguration The provider configuration
   * @param accessTokenToValidate       The token to evaluate or the expression to resolve in order to get the token value
   * @param scopesToValidate            The scopes to enforce when validating the token
   * @param resourceOwnerRoles          The resource owner roles to enforce when validating the token
   * @throws UnauthorizedTokenException if the token is not valid
   */
  @Throws(ValidateTokenErrorProvider.class)
  @MediaType(APPLICATION_JSON)
  public Result<String, Void> validateToken(@Config OAuth2ProviderConfiguration oAuth2ProviderConfiguration,
                                            AuthenticationHandler authenticationHandler,
                                            @Expression(REQUIRED) @Optional(
                                                defaultValue = "#[(attributes.headers['authorization'] splitBy ' ')[1]]") @Alias("accessToken") String accessTokenToValidate,
                                            @Expression(REQUIRED) @Optional @NullSafe @Alias("scopes") Set<String> scopesToValidate,
                                            @Expression(REQUIRED) @Optional @NullSafe Set<String> resourceOwnerRoles)
      throws UnauthorizedTokenException {

    //No token received
    if (accessTokenToValidate == null) {
      throw new UnauthorizedTokenException(createErrorMessage("No access token was received", UNAUTHORIZED.getStatusCode()));
    }

    TokenAuthentication.Builder tokenAuthenticationBuilder = TokenAuthentication.builder();

    tokenAuthenticationBuilder.withToken(accessTokenToValidate);

    //Validate ResourceOwnerRoles
    if (!resourceOwnerRoles.isEmpty()) {
      tokenAuthenticationBuilder.withResourceOwnerRoles(resourceOwnerRoles);
    }

    //Validate scopes
    if (!scopesToValidate.isEmpty()) {
      tokenAuthenticationBuilder.withScopes(scopesToValidate);
    }

    TokenAuthentication tokenAuthentication = tokenAuthenticationBuilder.build();
    try {
      //When setting the authentication, it will be validated by the authenticationHandler
      authenticationHandler.setAuthentication(asList(oAuth2ProviderConfiguration.getTokenSecurityProvider().getName()),
                                              tokenAuthentication);
    } catch (SecurityException e) {
      handleTokenAuthenticationException(e.getCause());
    } catch (SecurityProviderNotFoundException | UnknownAuthenticationTypeException e) {
      throw new ModuleException(OAUTH_SERVER_SECURITY, e);
    }

    return Result
        .<String, Void>builder()
        .output(buildJsonResponse((TokenAuthentication) authenticationHandler.getAuthentication().get()))
        .build();

  }

  private void handleTokenAuthenticationException(Throwable e) throws UnauthorizedTokenException {
    if (e instanceof UnauthorizedSecurityException) {
      throw new UnauthorizedTokenException(createErrorMessage(e.getMessage(), UNAUTHORIZED.getStatusCode()));
    } else if (e instanceof ForbiddenSecurityException) {
      throw new UnauthorizedTokenException(createErrorMessage(e.getMessage(), FORBIDDEN.getStatusCode()));
    } else {
      throw new ModuleException(OAUTH_SERVER_SECURITY, e);
    }
  }

  private String buildJsonResponse(TokenAuthentication authentication) {
    Token token = authentication.getTokenHolder().getAccessToken();
    JsonObject jsonObject = new JsonObject();
    jsonObject.addProperty("expires_in", token.getExpiresIn().getSeconds());
    jsonObject.addProperty("scope", stringifyScopes(token.getScopes()));

    if (!isBlank(token.getClientId())) {
      jsonObject.addProperty("client_id", token.getClientId());
    }

    ResourceOwnerAuthentication resourceOwnerAuthentication = authentication.getTokenHolder().getResourceOwnerAuthentication();
    if (resourceOwnerAuthentication != null) {
      String username = resourceOwnerAuthentication.getUsername();
      if (!isBlank(username)) {
        jsonObject.addProperty("username", username);
      }
    }

    return jsonObject.toString();
  }

  /**
   * Creates a new client and saves it in the configured client store.
   * <p/>
   *
   * {@sample.xml ../../../doc/oauth2-provider-connector.xml.sample oauth2-provider:create-client}
   *
   * @param clientId the Client Id
   * @param clientSecret the Client secret
   * @param clientType Clients can be PUBLIC or CONFIDENTIAL. If Confidential the secret is required. By default Clients are PUBLIC
   * @param clientName a friendly name for the Client
   * @param description a brief description of the Client
   * @param principal An optional principal to use when the ID can't be used with the security provider
   * @param redirectUris a list with the Client's valid redirect uris
   * @param authorizedGrantTypes the Client's supported grant types
   * @param scopes the Client's supported scopes
   * @param failIfPresent boolean to indicate if the operation should fail when the client id is already used.
   *                      Otherwise it should override the infotmation for the registered client
   */
  @Throws(CreateClientErrorProvider.class)
  public void createClient(@Config OAuth2ProviderConfiguration oAuth2ProviderConfiguration,
                           @Expression(SUPPORTED) String clientId,
                           @Expression(SUPPORTED) @Optional(defaultValue = "PUBLIC") @Alias("type") ClientType clientType,
                           @Expression(SUPPORTED) @Optional @Alias("secret") String clientSecret,
                           @Expression(SUPPORTED) @Optional String clientName,
                           @Expression(SUPPORTED) @Optional String description,
                           @Expression(SUPPORTED) @Optional String principal,
                           @Expression(REQUIRED) @Optional @NullSafe Set<String> redirectUris,
                           @Expression(REQUIRED) @Optional @NullSafe Set<RequestGrantType> authorizedGrantTypes,
                           @Expression(REQUIRED) @Optional @NullSafe Set<String> scopes,
                           @Expression(NOT_SUPPORTED) @Optional(defaultValue = "false") boolean failIfPresent)
      throws ClientAlreadyExistsException, OAuth2ConfigurationException {

    Client client = new Client(clientId, clientSecret, clientType, redirectUris, authorizedGrantTypes, scopes);
    client.setClientName(clientName);
    client.setDescription(description);
    client.setPrincipal(principal);

    try {
      if (CONFIDENTIAL.equals(clientType)
          && oAuth2ProviderConfiguration.getOAuthConfiguration().getClientSecurityProvider() == null) {
        checkArgument(!isEmpty(clientSecret),
                      format("Client secret should be specified for client: '%s' because his type is %s and no client security provider was configured",
                             clientName, CONFIDENTIAL.toString()));
      }
      oAuth2ProviderConfiguration.getClientManager().addClient(client, failIfPresent);
    } catch (IllegalArgumentException e) {
      throw new OAuth2ConfigurationException(e.getMessage());
    }
  }



  /**
   * <p>
   * Deletes a client from the store.
   * </p>
   * {@sample.xml ../../../doc/oauth2-provider-connector.xml.sample oauth2-provider:delete-client}
   *
   * @param clientId the Client Id
   * @throws NoSuchClientException If the client does not exist
   */
  @Throws(DeleteClientErrorProvider.class)
  public void deleteClient(@Config OAuth2ProviderConfiguration oAuth2ProviderConfiguration,
                           @Expression(SUPPORTED) String clientId)
      throws NoSuchClientException {
    oAuth2ProviderConfiguration.getClientManager().removeClient(clientId);
  }

  /**
   * <p>
   * Revokes an access token or refresh token, invalidating the related refresh token or access token as well. If client
   * credentials need to be validated the validateClient credential should be used before.
   * </p>
   *
   * {@sample.xml ../../../doc/oauth2-provider-connector.xml.sample oauth2-provider:revoke-token}
   *
   * @param oAuth2ProviderConfiguration this provider configuration
   * @param token the token to revoke, it can be an access token or a refresh token
   * @throws InvalidTokenException if the token is not valid
   */
  @Throws(RevokeTokenErrorProvider.class)
  public void revokeToken(@Config OAuth2ProviderConfiguration oAuth2ProviderConfiguration,
                          @Expression(SUPPORTED) final String token)
      throws InvalidTokenException {

    TokenStore tokenStore = oAuth2ProviderConfiguration.getTokenStore();
    AccessTokenStoreHolder accessToken = tokenStore.retrieveByAccessToken(token);
    if (accessToken != null) {
      tokenStore.remove(token);
      return;
    }
    AccessTokenStoreHolder refreshToken = tokenStore.retrieveByRefreshToken(token);
    if (refreshToken != null) {
      tokenStore.remove(refreshToken.getAccessToken().getAccessToken());
      return;
    }
    throw new InvalidTokenException("Token is invalid");
  }



  private Message createErrorMessage(String reasonPhrase, int statusCode) {
    return this.createErrorMessage(reasonPhrase, statusCode, null);
  }

  private Message createErrorMessage(String reasonPhrase, int statusCode, MultiMap<String, String> headers) {
    if (headers == null) {
      headers = new MultiMap<>();
    }

    headers.put(WWW_AUTHENTICATE, WWW_AUTHENTICATE_HEADER_VALUE);

    return Message.builder().nullValue().attributesValue(new HttpListenerResponseAttributes(statusCode,
                                                                                            reasonPhrase,
                                                                                            headers))
        .build();
  }

}
