/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal;

import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BASIC;
import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BEARER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.SCOPE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.UTF_8;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.INVALID_REQUEST;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.INVALID_SCOPE;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.TEMPORARILY_UNAVAILABLE;
import static java.lang.Boolean.getBoolean;
import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static org.apache.commons.collections.CollectionUtils.intersection;
import static org.apache.commons.collections.CollectionUtils.isEmpty;
import static org.apache.commons.collections.CollectionUtils.isNotEmpty;
import static org.apache.commons.collections.CollectionUtils.isSubCollection;
import static org.apache.commons.lang3.StringUtils.join;
import static org.apache.commons.lang3.StringUtils.replaceChars;
import static org.apache.commons.lang3.StringUtils.stripEnd;
import static org.apache.commons.lang3.StringUtils.trimToEmpty;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.core.api.config.MuleProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.core.api.util.Base64.DONT_BREAK_LINES;
import static org.mule.runtime.core.api.util.Base64.encodeBytes;
import static org.mule.runtime.core.api.util.Base64.decode;
import static org.mule.runtime.core.api.util.StringUtils.splitAndTrim;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ProviderGrantType.valueOf;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.core.api.util.StringUtils;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;

import com.mulesoft.modules.oauth2.provider.api.Constants.ProviderGrantType;
import com.mulesoft.modules.oauth2.provider.internal.processor.RequestData;
import com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class Utils {

  private static final Logger LOGGER = LoggerFactory.getLogger(Utils.class);
  private static final Charset DEFAULT_HTTP_ENCODING = Charset.forName("ISO-8859-1");

  private static final boolean strictContentType = getBoolean(SYSTEM_PROPERTY_PREFIX + "strictContentType");


  private Utils() {
    throw new UnsupportedOperationException("do not instantiate");
  }


  private static byte[] encodeBase64URLSafe(final byte[] binaryData) {
    try {
      return stripEnd(replaceChars(encodeBytes(binaryData, DONT_BREAK_LINES), "+/", "-_"), "=").getBytes();
    } catch (IOException e) {
      throw new MuleRuntimeException(createStaticMessage("Could not encode to Base64"), e);
    }
  }


  /*
   * Why not just a UUID? Read chapter 6 of http://www.ietf.org/rfc/rfc4122.txt
   */
  public static String generateUniqueId(final String salt) {
    checkArgument(!StringUtils.isEmpty(salt), "salt can't be empty");

    final byte[] saltyBytes = salt.getBytes();
    final UUID uuid = UUID.randomUUID();

    final ByteBuffer byteBuffer = ByteBuffer.allocate(saltyBytes.length + (2 * Long.SIZE / 8))
        .put(saltyBytes)
        .putLong(uuid.getMostSignificantBits())
        .putLong(uuid.getLeastSignificantBits());

    byte[] hashedBytes;
    try {
      MessageDigest sha512Hash = MessageDigest.getInstance("SHA-512");
      hashedBytes = sha512Hash.digest(byteBuffer.array());
    } catch (NoSuchAlgorithmException e) {
      throw new MuleRuntimeException(createStaticMessage("Could not generate unique Id"), e);
    }

    return new String(encodeBase64URLSafe(hashedBytes));
  }

  public static String generateUniqueId() {
    return generateUniqueId(Long.toString(Utils.class.hashCode()));
  }

  public static String extractCredentialsFromAuthorizationHeader(final String authorization,
                                                                 final String scheme,
                                                                 final String encoding) {
    final String[] tokens = splitAndTrim(authorization, " ");

    final boolean validHeader = tokens != null && tokens.length == 2
        && org.apache.commons.lang3.StringUtils.equals(tokens[0], scheme);

    if (!validHeader) {
      return null;
    }

    final String token = tokens[1];

    if (HTTP_AUTHORIZATION_SCHEME_BEARER.equals(scheme)) {
      return token;
    } else if (HTTP_AUTHORIZATION_SCHEME_BASIC.equals(scheme)) {
      return newString(decode(token), encoding);
    } else {
      LOGGER.warn("Unsupported authorization scheme: " + scheme);
      return null;
    }
  }

  private static String newString(final byte[] bytes, final String encoding) {
    try {
      return new String(bytes, encoding);
    } catch (final UnsupportedEncodingException uee) {
      throw new MuleRuntimeException(uee);
    }
  }

  public static Set<String> tokenize(final String scope) {
    return tokenize(scope, " ");
  }

  public static Set<String> tokenize(final String scope, String delimiter) {
    final String[] tokens = splitAndTrim(scope, delimiter);
    return tokens == null ? new HashSet<>() : new HashSet<>(asList(tokens));
  }

  public static String stringifyScopes(final Set<String> scopes) {
    return trimToEmpty(join(scopes, ' '));
  }

  public static Set<String> computeEffectiveScopeOrFail(final Set<String> requestedScopes,
                                                        final Set<String> clientScopes,
                                                        final Set<String> configuredScopes)
      throws RequestProcessingException {
    // effectively supported scopes is the intersection of the client ones
    // and the provider ones
    @SuppressWarnings("unchecked")
    final Collection<String> supportedScopes = intersection(configuredScopes, clientScopes);

    if (isEmpty(supportedScopes) && !isEmpty(configuredScopes)) {
      // the client doesn't authorize any of the configured scopes, log details
      // and return a temporary error without giving much details on the cause
      LOGGER.warn("Client configured with scopes: " + clientScopes
          + " doesn't match any provider-configured scopes: " + configuredScopes);
      throw new RequestProcessingException(TEMPORARILY_UNAVAILABLE,
                                           "Client configuration error");
    }

    if (isEmpty(requestedScopes) && isNotEmpty(supportedScopes)) {
      throw new RequestProcessingException(INVALID_REQUEST, "Missing mandatory parameter: "
          + SCOPE_PARAMETER);
    }

    if (isEmpty(requestedScopes)) {
      return emptySet();
    }

    if (isSubCollection(requestedScopes, supportedScopes)) {
      return requestedScopes;
    } else {
      // If we can't satisfy all the requested scopes, we fail. If this is
      // considered too strict, we could grant the intersection of
      // requestedScopes and supportedScopes.
      throw new RequestProcessingException(INVALID_SCOPE);
    }
  }

  public static String urlDecode(final String encoded) {
    if (encoded == null) {
      return null;
    }

    try {
      return URLDecoder.decode(encoded, UTF_8);
    } catch (final UnsupportedEncodingException uee) {
      throw new MuleRuntimeException(uee);
    }
  }

  public static Set<ProviderGrantType> parseProviderGrantTypes(final String providerGrantTypes) {
    final Set<ProviderGrantType> result = new HashSet<>();
    final Set<String> tokens = tokenize(providerGrantTypes, ",");

    if (tokens != null) {
      for (final String token : tokens) {
        result.add(valueOf(token));
      }
    }

    return result;
  }

  public static Charset resolveMessageEncoding(RequestData requestData) {
    String contentTypeValue =
        requestData == null ? null : requestData.getContext().getRequest().getHeaderValueIgnoreCase(CONTENT_TYPE);
    MediaType mediaType = getMediaType(contentTypeValue, DEFAULT_HTTP_ENCODING);
    return mediaType.getCharset().get();
  }

  private static MediaType getMediaType(final String contentTypeValue, Charset defaultCharset) {
    MediaType mediaType = MediaType.ANY;

    if (contentTypeValue != null) {
      try {
        mediaType = MediaType.parse(contentTypeValue);
      } catch (IllegalArgumentException e) {
        // need to support invalid Content-Types
        if (strictContentType) {
          throw e;
        } else {
          LOGGER.warn(format("%s when parsing Content-Type '%s': %s", e.getClass().getName(), contentTypeValue, e.getMessage()));
          LOGGER.warn(format("Using default encoding: %s", defaultCharset().name()));
        }
      }
    }
    if (!mediaType.getCharset().isPresent()) {
      return mediaType.withCharset(defaultCharset);
    } else {
      return mediaType;
    }
  }

  public static ResponseStatusCallback getResponseStatusCallback() {
    return new ResponseStatusCallback() {

      @Override
      public void responseSendFailure(Throwable throwable) {
        logFailureWhenSendingResponse(throwable);
      }

      @Override
      public void responseSendSuccessfully() {
        //Nothing to do
      }
    };
  }

  private static void logFailureWhenSendingResponse(Throwable throwable) {
    LOGGER.warn("Failure sending oauth2 provider static content response: " + throwable.getMessage());
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(throwable.getMessage(), throwable);
    }
  }

}
