/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal.processor;

import static com.mulesoft.modules.oauth2.provider.api.Constants.ACCESS_TOKEN_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.CODE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.EXPIRES_IN_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.TOKEN;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ResponseType.CODE;
import static com.mulesoft.modules.oauth2.provider.api.Constants.SCOPE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.TOKEN_TYPE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.ACCESS_DENIED;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.UNSUPPORTED_RESPONSE_TYPE;
import static org.apache.commons.collections.CollectionUtils.isNotEmpty;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ResponseType;
import org.mule.runtime.api.security.Authentication;
import org.mule.runtime.api.security.SecurityException;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;

import com.mulesoft.modules.oauth2.provider.api.AuthorizationRequest;
import com.mulesoft.modules.oauth2.provider.api.ResourceOwnerAuthentication;
import com.mulesoft.modules.oauth2.provider.internal.Utils;
import com.mulesoft.modules.oauth2.provider.api.client.Client;
import com.mulesoft.modules.oauth2.provider.internal.config.OAuthConfiguration;
import com.mulesoft.modules.oauth2.provider.api.token.Token;

import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;

public class AuthorizationRequestProcessor extends OAuth2ProviderRequestProcessor {

  public AuthorizationRequestProcessor(final OAuthConfiguration configuration) {
    super(configuration);
  }

  public void processRequest(final RequestData requestData, final HttpResponseBuilder httpResponseBuilder)
      throws SecurityException {
    final ResponseType responseType = getSupportedResponseTypeOrFail(requestData);

    final Client client = getKnownClientOrFail(requestData);

    final Pair<Boolean, ResourceOwnerAuthentication> resourceOwnerAuthenticationResult = validateResourceOwnerCredentials(
                                                                                                                          client,
                                                                                                                          requestData);
    if (!resourceOwnerAuthenticationResult.getLeft()) {
      throw new RequestProcessingException(ACCESS_DENIED);
    }

    final String redirectUri = getValidRedirectionUriOrFail(client, requestData);

    final AuthorizationRequest authorizationRequest = new AuthorizationRequest(client.getClientId(),
                                                                               responseType, redirectUri,
                                                                               resourceOwnerAuthenticationResult.getRight());

    final Set<String> effectiveScopes = getEffectiveScopes(requestData, client);
    if (isNotEmpty(effectiveScopes)) {
      authorizationRequest.getScopes().addAll(effectiveScopes);
    }

    if (responseType == CODE) {
      processCodeRequest(authorizationRequest, requestData, httpResponseBuilder);
    } else if (responseType == ResponseType.TOKEN) {
      processTokenRequest(resourceOwnerAuthenticationResult.getRight(), authorizationRequest, requestData, httpResponseBuilder);
    } else {
      throw new RequestProcessingException(UNSUPPORTED_RESPONSE_TYPE,
                                           "Unsupported response type: " + responseType);
    }
  }

  private void processCodeRequest(final AuthorizationRequest authorizationRequest,
                                  final RequestData requestData,
                                  final HttpResponseBuilder httpResponseBuilder)
      throws SecurityException {
    final String authorizationGrant = configuration.getAuthorizationCodeManager()
        .generateAuthorizationCode(authorizationRequest);

    final String actualRedirectUri = buildRedirectUri(authorizationRequest.getRedirectUri(), requestData,
                                                      CODE_PARAMETER, authorizationGrant);

    setRedirectResponse(httpResponseBuilder, actualRedirectUri);
  }

  private void processTokenRequest(final ResourceOwnerAuthentication resourceOwnerAuthentication,
                                   final AuthorizationRequest authorizationRequest,
                                   final RequestData requestData,
                                   final HttpResponseBuilder httpResponseBuilder)
      throws RequestProcessingException {
    // implicit grant tokens are never refreshable
    final Token accessToken = configuration.getTokenManager().grantAccessToken(TOKEN,
                                                                               authorizationRequest,
                                                                               resourceOwnerAuthentication);

    final String actualRedirectUri = buildRedirectUri(authorizationRequest.getRedirectUri(), requestData,
                                                      ACCESS_TOKEN_PARAMETER, accessToken.getAccessToken(),
                                                      TOKEN_TYPE_PARAMETER, accessToken.getType(),
                                                      EXPIRES_IN_PARAMETER,
                                                      Long.toString(configuration.getTokenConfig().getTokenTtlInSeconds()),
                                                      SCOPE_PARAMETER,
                                                      Utils.stringifyScopes(authorizationRequest.getScopes()));

    setRedirectResponse(httpResponseBuilder, actualRedirectUri);
  }
}
