/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal.processor;

import static com.mulesoft.modules.oauth2.provider.api.Constants.CLIENT_ID_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.CLIENT_SECRET_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ERROR_DESCRIPTION_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ERROR_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.GRANT_TYPE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BASIC;
import static com.mulesoft.modules.oauth2.provider.api.Constants.PASSWORD_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.REDIRECT_URI_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RESPONSE_TYPE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ResponseType.TOKEN;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ResponseType.valueOfIgnoreCase;
import static com.mulesoft.modules.oauth2.provider.api.Constants.SCOPE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.STATE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.USERNAME_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.internal.Utils.computeEffectiveScopeOrFail;
import static com.mulesoft.modules.oauth2.provider.internal.Utils.resolveMessageEncoding;
import static com.mulesoft.modules.oauth2.provider.internal.Utils.tokenize;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.INVALID_CLIENT_ID;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.INVALID_GRANT;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.INVALID_REDIRECTION_URI;
import static com.mulesoft.modules.oauth2.provider.api.ratelimit.RateLimiter.Operation.RESOURCE_OWNER_LOGIN;
import static com.mulesoft.modules.oauth2.provider.api.ratelimit.RateLimiter.Outcome.FAILURE;
import static com.mulesoft.modules.oauth2.provider.api.ratelimit.RateLimiter.Outcome.SUCCESS;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.INVALID_REQUEST;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.RATE_LIMIT_EXCEEDED;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.SERVER_ERROR;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.UNAUTHORIZED_CLIENT;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.UNSUPPORTED_GRANT_TYPE;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.UNSUPPORTED_RESPONSE_TYPE;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingExceptionFactory.noClientAuthenticationException;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ArrayUtils.addAll;
import static org.apache.commons.lang3.StringUtils.equalsIgnoreCase;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.join;
import static org.apache.commons.lang3.StringUtils.stripToEmpty;
import static org.apache.commons.lang3.StringUtils.stripToNull;
import static org.apache.commons.lang3.StringUtils.substringAfter;
import static org.apache.commons.lang3.StringUtils.substringBefore;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.core.api.util.ExceptionUtils.extractCauseOfType;
import static org.mule.runtime.extension.api.annotation.param.MediaType.APPLICATION_JSON;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.BAD_REQUEST;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.MOVED_TEMPORARILY;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.TOO_MANY_REQUESTS;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.LOCATION;
import static org.mule.runtime.http.api.HttpHeaders.Values.APPLICATION_X_WWW_FORM_URLENCODED;
import static org.mule.runtime.http.api.utils.HttpEncoderDecoderUtils.decodeUrlEncodedBody;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.security.Credentials;
import org.mule.runtime.api.security.DefaultMuleAuthentication;
import org.mule.runtime.api.security.SecurityException;
import org.mule.runtime.core.api.security.DefaultMuleCredentials;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.runtime.http.api.domain.request.HttpRequestContext;

import com.mulesoft.modules.oauth2.provider.api.Constants.ResponseType;
import com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType;
import com.mulesoft.modules.oauth2.provider.api.ResourceOwnerAuthentication;
import com.mulesoft.modules.oauth2.provider.internal.Utils;
import com.mulesoft.modules.oauth2.provider.api.client.Client;
import com.mulesoft.modules.oauth2.provider.api.client.NoSuchClientException;
import com.mulesoft.modules.oauth2.provider.internal.config.OAuthConfiguration;
import com.mulesoft.modules.oauth2.provider.internal.ratelimit.RateLimitExceededException;
import com.mulesoft.modules.oauth2.provider.internal.token.InvalidGrantException;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;

public class OAuth2ProviderRequestProcessor {

  private static final Logger LOGGER = getLogger(OAuth2ProviderRequestProcessor.class);

  protected OAuthConfiguration configuration;

  public OAuth2ProviderRequestProcessor(final OAuthConfiguration configuration) {
    this.configuration = configuration;
  }

  public final void process(final HttpRequestContext httpRequestContext, HttpResponseBuilder httpResponseBuilder,
                            OAuth2ProviderProcessor oAuth2ProviderProcessor) {
    RequestData requestData = null;
    try {
      requestData = new RequestData(httpRequestContext);
      oAuth2ProviderProcessor.process(requestData, httpResponseBuilder);
    } catch (final RequestProcessingException rpe) {
      handleException(rpe, requestData, httpResponseBuilder);
    } catch (final Exception e) {
      handleException(convertToRequestProcessingException(e), requestData, httpResponseBuilder);
    }
  }

  protected RequestProcessingException convertToRequestProcessingException(final Exception e) {
    if (e instanceof InvalidGrantException) {
      return new RequestProcessingException(INVALID_GRANT, e.getMessage());
    } else if (e instanceof NoSuchClientException) {
      return new RequestProcessingException(UNAUTHORIZED_CLIENT, "Invalid client");
    } else if (e instanceof RateLimitExceededException) {
      return new RequestProcessingException(RATE_LIMIT_EXCEEDED);
    }

    final Throwable requestProcessingException = extractCauseOfType(e, RequestProcessingException.class).orElse(null);

    if (requestProcessingException != null) {
      return (RequestProcessingException) requestProcessingException;
    }

    final Throwable illegalArgumentException = extractCauseOfType(e, IllegalArgumentException.class).orElse(null);

    if (illegalArgumentException != null) {
      return new RequestProcessingException(INVALID_REQUEST, illegalArgumentException.getMessage());
    } else {
      return new RequestProcessingException(SERVER_ERROR, e);
    }
  }

  protected void handleException(final RequestProcessingException exception,
                                 final RequestData requestData, HttpResponseBuilder httpResponseBuilder) {
    if (exception.getErrorType() == SERVER_ERROR) {
      LOGGER.error("Unexpected exception", exception);
      httpResponseBuilder.statusCode(INTERNAL_SERVER_ERROR.getStatusCode());
      httpResponseBuilder.reasonPhrase("SERVER ERROR");
      setResponsePayload(httpResponseBuilder, resolveMessageEncoding(requestData).name(), ERROR_PARAMETER,
                         SERVER_ERROR.getErrorCode());
      return;
    } else if (exception.getErrorType() == RATE_LIMIT_EXCEEDED) {
      httpResponseBuilder.statusCode(TOO_MANY_REQUESTS.getStatusCode());
      httpResponseBuilder.entity(new EmptyHttpEntity());
      return;
    }

    final String[] parameters =
        {ERROR_PARAMETER, exception.getErrorType().getErrorCode(), ERROR_DESCRIPTION_PARAMETER, exception.getMessage()};

    final String redirectUri = getParameterFromBodyOrQuery(requestData, REDIRECT_URI_PARAMETER);

    boolean redirectedForError = false;
    if (isRedirectingForError(exception.getErrorType(), redirectUri)) {
      try {
        try {
          getSupportedResponseTypeOrFail(requestData);
        } catch (RequestProcessingException e) {
          //If redirectUri will fail due to a wrong, missing or duplicate response type,
        }
        final String actualRedirectUri = buildErrorResponseRedirectUri(redirectUri, requestData, parameters);
        setRedirectResponse(httpResponseBuilder, actualRedirectUri);
        redirectedForError = true;
      } catch (final RequestProcessingException rpe) {
        //Do nothing, respond 400
      }
    }

    // error must be replied direct
    if (!redirectedForError) {
      httpResponseBuilder.statusCode(BAD_REQUEST.getStatusCode());
      setResponsePayload(httpResponseBuilder, resolveMessageEncoding(requestData).name(), parameters);
    }

  }

  protected boolean isRedirectingForError(final RequestProcessingException.ErrorType errorType, final String redirectUri) {
    return errorType.isDoRedirect() && isNotBlank(redirectUri);
  }

  protected void setResponsePayload(HttpResponseBuilder responseBuilder, final String encoding, final String... parameters) {
    responseBuilder.entity(new ByteArrayHttpEntity(buildEncodedParameters(encoding, parameters).getBytes()));
    responseBuilder.addHeader(CONTENT_TYPE, APPLICATION_X_WWW_FORM_URLENCODED.toRfcString());
  }

  protected void setRedirectResponse(final HttpResponseBuilder httpResponseBuilder, final String actualRedirectUri) {
    httpResponseBuilder.statusCode(MOVED_TEMPORARILY.getStatusCode());
    httpResponseBuilder.addHeader(CONTENT_TYPE, APPLICATION_X_WWW_FORM_URLENCODED.toRfcString());
    httpResponseBuilder.addHeader(LOCATION, actualRedirectUri);
  }

  protected String buildErrorResponseRedirectUri(String redirectUri, RequestData requestData, String... parameters)
      throws RequestProcessingException {
    try {
      return buildRedirectUri(redirectUri, requestData, parameters);
    } catch (RequestProcessingException e) {
      return buildRedirectUri(redirectUri, requestData, false, parameters);
    }
  }

  protected String buildRedirectUri(final String redirectUri,
                                    final RequestData requestData,
                                    final String... parameters)
      throws RequestProcessingException {


    final ResponseType responseType = getSupportedResponseTypeOrFail(requestData);
    final boolean parametersInFragment = responseType == TOKEN;
    return buildRedirectUri(redirectUri, requestData, parametersInFragment, parameters);
  }

  private String buildRedirectUri(final String redirectUri,
                                  final RequestData requestData,
                                  boolean parametersInFragment,
                                  final String... parameters)
      throws RequestProcessingException {
    // automatic state injection
    final String state = getOptionalParameter(requestData, STATE_PARAMETER);
    String encoding = resolveMessageEncoding(requestData).name();
    if (isNotBlank(state)) {
      return buildRedirectUri(redirectUri, encoding, parametersInFragment,
                              (String[]) addAll(parameters, new String[] {STATE_PARAMETER, state}));
    } else {
      return buildRedirectUri(redirectUri, encoding, parametersInFragment, parameters);
    }
  }

  public String buildRedirectUri(final String redirectUri,
                                 final String encoding,
                                 final boolean parametersInFragment,
                                 final String... parameters)
      throws RequestProcessingException {
    final String encodedParameters = buildEncodedParameters(encoding, parameters);

    final URI uri = newURI(redirectUri);
    final String noFragmentRedirectUri = stripFragment(uri);

    if (isBlank(encodedParameters)) {
      return noFragmentRedirectUri;
    }

    final String parametersSeparator = parametersInFragment ? "#" : (hasQuery(uri) ? "&" : "?");

    return noFragmentRedirectUri + parametersSeparator + encodedParameters;
  }

  private boolean hasQuery(final URI uri) {
    return isNotBlank(uri.getQuery());
  }

  protected Map<String, Object> keyValuePairsToMap(final Object... parameters) {
    checkArgument(parameters.length % 2 == 0, "need an even number of (param name, param value) string pairs");

    final Map<String, Object> result = new HashMap<String, Object>();
    for (int i = 0; i < parameters.length; i += 2) {
      final Object value = parameters[i + 1];
      if (value == null) {
        continue;
      }
      final String key = (String) parameters[i];
      result.put(key, value);
    }
    return result;
  }

  private String buildEncodedParameters(final String encoding, final String... parameters) {
    checkArgument(parameters.length % 2 == 0, "need an even number of (param name, param value) string pairs");

    final StringBuilder parametersBuilder = new StringBuilder();

    for (int i = 0; i < parameters.length; i += 2) {
      final String name = parameters[i];
      final String value = parameters[i + 1];

      if (isNotBlank(value)) {
        if (parametersBuilder.length() > 0) {
          parametersBuilder.append('&');
        }

        parametersBuilder.append(name).append('=').append(urlEncode(value, encoding));
      }
    }
    return parametersBuilder.toString();
  }

  protected String getMandatoryParameterOrFail(final RequestData requestData, final String parameterName)
      throws RequestProcessingException {
    final String parameterValue = getOptionalParameter(requestData, parameterName);

    if (isBlank(parameterValue)) {
      RequestProcessingException.ErrorType errorType = RequestProcessingException.ErrorType.findByParameterName(parameterName);
      if (errorType == null) {
        errorType = INVALID_REQUEST;
      }

      throw new RequestProcessingException(errorType, "Missing mandatory parameter: " + parameterName);
    }

    return parameterValue;
  }

  protected String getOptionalParameter(final RequestData requestData, final String parameterName) {
    final String parameterValue = getParameterFromBodyOrQuery(requestData, parameterName);

    if (isBlank(parameterValue)) {
      return null;
    }

    return parameterValue;
  }

  protected ResponseType getSupportedResponseTypeOrFail(final RequestData requestData)
      throws RequestProcessingException {
    final String responseTypeString = getMandatoryParameterOrFail(requestData,
                                                                  RESPONSE_TYPE_PARAMETER);
    try {
      final ResponseType responseType = valueOfIgnoreCase(responseTypeString);
      if (!configuration.isAuthorizationResponseTypeSupported(responseType)) {
        throw new RequestProcessingException(UNSUPPORTED_RESPONSE_TYPE,
                                             buildUnsupportedResponseTypeErrorMessage(responseTypeString));
      }
      return responseType;
    } catch (final IllegalArgumentException iae) {
      throw new RequestProcessingException(UNSUPPORTED_RESPONSE_TYPE,
                                           buildUnsupportedResponseTypeErrorMessage(responseTypeString));
    }
  }

  private String buildUnsupportedResponseTypeErrorMessage(final String responseTypeString) {
    return "Response type '" + responseTypeString + "' is not supported";
  }

  protected RequestGrantType getSupportedRequestGrantTypeOrFail(final RequestData requestData)
      throws RequestProcessingException {
    final String grantType = getMandatoryParameterOrFail(requestData, GRANT_TYPE_PARAMETER);
    try {
      final RequestGrantType requestGrantType = RequestGrantType.valueOfIgnoreCase(grantType);
      if (!configuration.isRequestGrantTypeSupported(requestGrantType)) {
        throw new RequestProcessingException(UNSUPPORTED_GRANT_TYPE,
                                             buildUnsupportedRequestGrantTypeErrorMessage(grantType));
      }
      return requestGrantType;
    } catch (final IllegalArgumentException iae) {
      throw new RequestProcessingException(UNSUPPORTED_GRANT_TYPE,
                                           buildUnsupportedRequestGrantTypeErrorMessage(grantType));
    }
  }

  private String buildUnsupportedRequestGrantTypeErrorMessage(final String grantType) {
    return "Grant type '" + grantType + "' is not supported";
  }

  protected Client getKnownClientOrFail(final RequestData requestData) throws SecurityException {
    final Credentials credentials = extractClientCredentials(requestData);
    final String clientId = credentials.getUsername();

    return configuration.getClientManager().getClientById(clientId);
  }

  protected String getValidRedirectionUriOrFail(final Client client, final RequestData requestData)
      throws RequestProcessingException {
    final String redirectUri = getMandatoryParameterOrFail(requestData, REDIRECT_URI_PARAMETER);
    if (!client.isValidRedirectUri(redirectUri)) {
      throw new RequestProcessingException(INVALID_REDIRECTION_URI);
    } else {
      return redirectUri;
    }
  }

  private String getParameterFromBodyOrQuery(final RequestData requestData, final String parameterName) {
    String parameterValue = null;
    List<String> parameterRawValue = null;

    // try first as a POST body param
    String requestContentType = requestData.getContext().getRequest().getHeaderValueIgnoreCase(CONTENT_TYPE);
    if (requestContentType != null
        && requestContentType.contains((APPLICATION_X_WWW_FORM_URLENCODED.withoutParameters().toRfcString()))) {
      Map<String, String> parameters = decodeUrlEncodedBody(requestData.getContent(), resolveMessageEncoding(requestData));
      parameterRawValue = parameters.entrySet().stream().filter(entry -> equalsIgnoreCase(entry.getKey(), parameterName))
          .map(Map.Entry::getValue).collect(toList());
      if (!parameterRawValue.isEmpty()) {
        parameterValue = getSingleParameterValue(parameterRawValue);
      }
    }

    // try to read from query parameter map (in the new HTTP module query params are no longer inbound properties)
    if (isBlank(parameterValue)) {
      parameterValue = getSingleParameterValue(requestData.getContext().getRequest().getQueryParams().getAll(parameterName));
    }

    if (isBlank(parameterValue)) {
      parameterValue = requestData.getContext().getRequest().getHeaderValueIgnoreCase(parameterName);
    }
    return stripToNull(parameterValue);
  }



  private String getSingleParameterValue(final Object values) {
    if (values == null) {
      return null;
    }

    if (values instanceof String) {
      return (String) values;
    } else if (values instanceof List<?>) {
      return join(new HashSet<Object>((List<?>) values), ' ');
    } else if (values.getClass().isArray()) {
      return join(new HashSet<Object>(Arrays.asList((Object[]) values)), ' ');
    } else {
      return values.toString();
    }
  }

  private String urlEncode(final String value, final String encoding) {
    try {
      return URLEncoder.encode(value, encoding);
    } catch (final UnsupportedEncodingException uee) {
      throw new MuleRuntimeException(uee);
    }
  }

  private String urlDecode(final String value, final String encoding) {
    try {
      return URLDecoder.decode(value, encoding);
    } catch (final UnsupportedEncodingException uee) {
      throw new MuleRuntimeException(uee);
    }
  }

  private URI newURI(final String uri) throws RequestProcessingException {
    try {
      return new URI(uri);
    } catch (final URISyntaxException urie) {
      throw new RequestProcessingException(INVALID_REDIRECTION_URI, urie.getMessage());
    }
  }

  private String stripFragment(final URI uri) {
    final String fragment = uri.getFragment();
    if (isNotBlank(fragment)) {
      return substringBefore(uri.toString(), "#" + fragment);
    } else {
      return uri.toString();
    }
  }

  protected Credentials extractResourceOwnerCredentials(final RequestData requestData)
      throws RequestProcessingException {
    final String username = getMandatoryParameterOrFail(requestData, USERNAME_PARAMETER);
    final String password = stripToEmpty(getOptionalParameter(requestData, PASSWORD_PARAMETER));
    return new DefaultMuleCredentials(username, password.toCharArray());
  }

  protected Credentials extractClientCredentials(final RequestData requestData) throws RequestProcessingException {
    String clientId = getOptionalParameter(requestData, CLIENT_ID_PARAMETER);
    String clientSecret = stripToEmpty(getOptionalParameter(requestData,
                                                            CLIENT_SECRET_PARAMETER));
    String basicAuthHeader = getOptionalParameter(requestData, AUTHORIZATION);
    if (isBlank(basicAuthHeader)) {
      basicAuthHeader = requestData.getContext().getRequest().getHeaderValueIgnoreCase(AUTHORIZATION);
    }

    if (isBlank(clientId) && isBlank(basicAuthHeader)) {
      throw noClientAuthenticationException();
    }

    if (isNotBlank(clientSecret) && isNotBlank(basicAuthHeader)) {
      throw new RequestProcessingException(INVALID_REQUEST,
                                           "Multiple client authentications found");
    }

    if (isBlank(basicAuthHeader)) {
      return new ClientSecretCredentials(clientId, clientSecret.toCharArray());
    } else {
      final Pair<String, String> idAndSecret = getBasicAuthClientIdAndSecret(requestData);
      clientId = idAndSecret.getLeft();
      clientSecret = idAndSecret.getRight();

      if (isBlank(clientId)) {
        throw new RequestProcessingException(INVALID_REQUEST, "Invalid '"
            + AUTHORIZATION
            + "' header");
      }

      return new ClientSecretCredentials(clientId, clientSecret.toCharArray());
    }
  }

  private Pair<String, String> getBasicAuthClientIdAndSecret(final RequestData requestData) {
    final String basicAuthHeader = requestData.getContext().getRequest().getHeaderValueIgnoreCase(AUTHORIZATION);

    final String decodedBasicAuthHeader = Utils.extractCredentialsFromAuthorizationHeader(
                                                                                          basicAuthHeader,
                                                                                          HTTP_AUTHORIZATION_SCHEME_BASIC,
                                                                                          resolveMessageEncoding(requestData)
                                                                                              .name());
    // as per spec, username and password are URL encoded:
    // http://tools.ietf.org/html/draft-ietf-oauth-v2-31#2.3.1
    final String clientId = Utils.urlDecode(substringBefore(decodedBasicAuthHeader, ":"));
    final String clientSecret = stripToEmpty(Utils.urlDecode(substringAfter(decodedBasicAuthHeader, ":")));

    return Pair.of(clientId, clientSecret);
  }

  protected Set<String> getEffectiveScopes(final RequestData requestData, final Client client)
      throws RequestProcessingException {
    final Set<String> requestedScopes = tokenize(getOptionalParameter(requestData, SCOPE_PARAMETER), " ");

    // If there are no scopes defined in the client, then use the default scopes if configured.
    Set<String> clientScopes = client.getScopes();
    if (clientScopes.isEmpty() && !configuration.getDefaultScopes().isEmpty()) {
      clientScopes = configuration.getDefaultScopes();
    }

    return computeEffectiveScopeOrFail(requestedScopes, clientScopes, configuration.getSupportedScopes());
  }

  protected Pair<Boolean, ResourceOwnerAuthentication> validateResourceOwnerCredentials(final Client client,
                                                                                        final RequestData requestData)
      throws SecurityException {
    ResourceOwnerAuthentication authentication = null;

    final Credentials credentials = extractResourceOwnerCredentials(requestData);

    configuration.getRateLimiter().checkOperationAuthorized(RESOURCE_OWNER_LOGIN,
                                                            credentials.getUsername());

    try {
      authentication = configuration.getResourceOwnerSecurityProvider().authenticate(new DefaultMuleAuthentication(credentials));
    } catch (final Exception e) {
      logValidationException(client, credentials, e);
    }
    boolean success = false;
    if (authentication != null) {
      success = true;
    }

    configuration.getRateLimiter().recordOperationOutcome(RESOURCE_OWNER_LOGIN,
                                                          credentials.getUsername(), success ? SUCCESS : FAILURE);

    return Pair.of(success, authentication);
  }

  private void logValidationException(Client client, Credentials credentials, Exception e) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.warn(getValidationMessage(client, credentials), e);
    } else {
      LOGGER.warn(getValidationMessage(client, credentials));
    }
  }

  private String getValidationMessage(Client client, Credentials credentials) {
    return format("Failed to validate client credentials for client ID: %s and principal: %s", client.getClientId(),
                  credentials.getUsername());
  }

  protected boolean validateClientCredentials(final Client client, final RequestData requestData)
      throws RequestProcessingException {
    final Credentials credentials = extractClientCredentials(requestData);

    boolean validated = client.isAuthenticatedBy((ClientSecretCredentials) credentials);

    if (validated) {
      return true;
    }
    //Else, try to validate it with the security provider

    if (configuration.getClientSecurityProvider() == null) {
      LOGGER.warn("Client ID: "
          + client.getClientId()
          + " failed to present a secret and no security provider is configured to validate its credentials");
      return false;
    }

    // allow using a custom security principal for the cases when the security
    // provider doesn't use client_id as its natural principal but something else
    final String effectivePrincipal = isNotBlank(client.getPrincipal())
        ? client.getPrincipal()
        : credentials.getUsername();
    final Credentials effectiveCredentials = new DefaultMuleCredentials(effectivePrincipal,
                                                                        credentials.getPassword());

    try {
      configuration.getClientSecurityProvider().authenticate(new DefaultMuleAuthentication(effectiveCredentials));
      return true;
    } catch (final Exception e) {
      logValidationException(client, effectiveCredentials, e);
      return false;
    }
  }

  protected void failIfParameterPresentMultipleTimes(RequestData requestData, String... parameterNames)
      throws RequestProcessingException {
    for (String parameterName : parameterNames) {
      List<String> asBody =
          decodeUrlEncodedBody(requestData.getContent(), resolveMessageEncoding(requestData)).getAll(parameterName);
      Collection<String> asHeader = requestData.getContext().getRequest().getHeaderValuesIgnoreCase(parameterName);
      List<String> asQueryParam = requestData.getContext().getRequest().getQueryParams().getAll(parameterName);
      if ((asBody.size() + asHeader.size() + asQueryParam.size()) > 1) {
        throw new RequestProcessingException(INVALID_REQUEST, format("Found multiple values for parameter: %s", parameterName));
      }
    }
  }

}
