/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal.security;

import org.mule.runtime.api.security.Authentication;
import org.mule.runtime.api.security.SecurityException;
import org.mule.runtime.core.api.security.SecurityProvider;

import com.mulesoft.modules.oauth2.provider.api.ResourceOwnerAuthentication;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.ldap.userdetails.LdapUserDetails;

/**
 * ResourceOwnerSecurityProvider used for when the Mule Spring Module is present.
 */
public class SpringAwareResourceOwnerSecurityProvider extends ResourceOwnerSecurityProvider {

  private List<UserDetailsService> userDetailsServices;

  public SpringAwareResourceOwnerSecurityProvider(SecurityProvider delegate, List<UserDetailsService> userDetailsServices) {
    super(delegate);
    this.userDetailsServices = userDetailsServices;
  }

  @Override
  public ResourceOwnerAuthentication authenticate(Authentication authentication) throws SecurityException {
    Authentication authenticatedAuth = super.authenticate(authentication);
    String authenticatedAuthUsername = getUsername(authenticatedAuth);
    Set<String> authenticatedAuthRoles = getRoles(authenticatedAuthUsername);
    return new ResourceOwnerAuthentication.Builder()
        .withPrincipal(authenticatedAuth.getPrincipal())
        .withUsername(authenticatedAuthUsername)
        //After authentication, never save credentials
        .withCredentials(null)
        .withRoles(authenticatedAuthRoles)
        .withProperties(authenticatedAuth.getProperties())
        .build();
  }

  private String getUsername(Authentication authentication) {
    Object user = authentication.getPrincipal();

    if (user instanceof User) {
      return ((User) user).getUsername();
    } else if (user instanceof LdapUserDetails) {
      return ((LdapUserDetails) user).getUsername();
    } else {
      return (String) user;
    }
  }

  /**
   * Gets the roles from the username of the Authentication when it handled by spring-module.
   * This is a little bit of a hack needed because SpringAuthenticationAdapter is an internal class of the spring-module and
   * the method getAuthorities() can't be called.
   * @param username the username defined for the authentication
   * @return roles associated with the authentication
   */
  private Set<String> getRoles(String username) {
    Set<String> roles = new HashSet<>();
    UserDetails userDetails = null;
    for (UserDetailsService userDetailsService : userDetailsServices) {
      try {
        userDetails = userDetailsService.loadUserByUsername(username);
      } catch (UsernameNotFoundException e) {
        //Do nothing, keep looking
      }
      if (userDetails != null) {
        for (final GrantedAuthority authority : userDetails.getAuthorities()) {
          roles.add(authority.getAuthority());
        }
      }
    }
    return roles;
  }

}
