/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal.token;

import static org.mule.runtime.api.util.Preconditions.checkArgument;
import org.mule.runtime.api.security.Authentication;

import com.mulesoft.modules.oauth2.provider.api.token.AccessTokenStoreHolder;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Class used to store information used in a token authentication. When wanting to authenticate an inbound token,
 * an instance of this class should be sent to the {@link org.mule.runtime.core.api.security.SecurityProvider} to authenticate.
 * <p/>
 */
public class TokenAuthentication implements Authentication {

  private String token;
  private Map<String, Object> properties;
  private Set<String> scopes;
  private Set<String> resourceOwnerRoles;
  private AccessTokenStoreHolder tokenHolder;

  private TokenAuthentication(Builder builder) {
    this.token = builder.token;
    this.tokenHolder = builder.tokenHolder;
    this.scopes = builder.scopes;
    this.resourceOwnerRoles = builder.resourceOwnerRoles;
    this.properties = builder.properties;
  }

  public static Builder builder() {
    return new Builder();
  }

  public static Builder builder(TokenAuthentication tokenAuthentication) {
    return new Builder()
        .withScopes(tokenAuthentication.getScopes())
        .withResourceOwnerRoles(tokenAuthentication.getResourceOwnerRoles())
        .withProperties(tokenAuthentication.getProperties())
        .withToken(tokenAuthentication.getToken())
        .withTokenHolder(tokenAuthentication.getTokenHolder());
  }

  @Override
  public Object getCredentials() {
    //Always return empty because token authentication never has credentials associated with it
    return null;
  }

  @Override
  public Object getPrincipal() {
    //There is also no principal for a token.
    return null;
  }

  @Override
  public Map<String, Object> getProperties() {
    return properties;
  }

  @Override
  public Authentication setProperties(Map<String, Object> properties) {
    return new Builder().withToken(token).withTokenHolder(tokenHolder).withScopes(scopes)
        .withResourceOwnerRoles(resourceOwnerRoles).withProperties(properties).build();
  }

  public String getToken() {
    return token;
  }

  public Set<String> getScopes() {
    return scopes;
  }

  public Set<String> getResourceOwnerRoles() {
    return resourceOwnerRoles;
  }

  public AccessTokenStoreHolder getTokenHolder() {
    return tokenHolder;
  }

  public static class Builder {

    private String token;
    private AccessTokenStoreHolder tokenHolder;
    private Set<String> scopes;
    private Set<String> resourceOwnerRoles;
    private Map<String, Object> properties = new HashMap<>();

    public Builder withToken(String token) {
      this.token = token;
      return this;
    }

    public Builder withProperties(Map<String, Object> properties) {
      this.properties = properties;
      return this;
    }

    public Builder withScopes(Set<String> scopes) {
      this.scopes = scopes;
      return this;
    }

    public Builder withResourceOwnerRoles(Set<String> resourceOwnerRoles) {
      this.resourceOwnerRoles = resourceOwnerRoles;
      return this;
    }

    public Builder withTokenHolder(AccessTokenStoreHolder tokenHolder) {
      this.tokenHolder = tokenHolder;
      return this;
    }

    public TokenAuthentication build() {
      checkArgument(token != null, "Token must not be null in TokenAuthentication");
      return new TokenAuthentication(this);
    }
  }

}
