/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal.token;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.core.api.util.StringUtils.isEmpty;

import org.mule.runtime.api.security.Authentication;

import com.mulesoft.modules.oauth2.provider.api.ResourceOwnerAuthentication;
import com.mulesoft.modules.oauth2.provider.api.token.AccessTokenStoreHolder;
import com.mulesoft.modules.oauth2.provider.api.token.Token;
import com.mulesoft.modules.oauth2.provider.api.token.TokenStore;
import com.mulesoft.modules.oauth2.provider.api.AuthorizationRequest;
import com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType;
import com.mulesoft.modules.oauth2.provider.api.token.generator.RefreshTokenStrategy;
import com.mulesoft.modules.oauth2.provider.api.token.generator.TokenGeneratorStrategy;

import java.util.Set;
import java.util.concurrent.TimeUnit;


public class TokenManager {

  private final TokenStore tokenStore;
  private final TokenGeneratorStrategy tokenGeneratorStrategy;
  private final RefreshTokenStrategy refreshTokenStrategy;
  private final long tokenExpirationInterval;
  private final TimeUnit tokenExpirationIntervalTimeUnit;

  public TokenManager(final TokenStore tokenStore,
                      final TokenGeneratorStrategy tokenGeneratorStrategy,
                      final RefreshTokenStrategy refreshTokenStrategy,
                      long tokenExpirationInterval,
                      TimeUnit tokenExpirationIntervalTimeUnit) {
    checkArgument(tokenStore != null, "tokenStore can't be null");
    checkArgument(tokenGeneratorStrategy != null, "tokenGeneratorStrategy can't be null");
    checkArgument(tokenExpirationIntervalTimeUnit != null, "token expiration interval time unit can't be null");

    this.tokenStore = tokenStore;
    this.tokenGeneratorStrategy = tokenGeneratorStrategy;
    this.refreshTokenStrategy = refreshTokenStrategy;
    this.tokenExpirationInterval = tokenExpirationInterval;
    this.tokenExpirationIntervalTimeUnit = tokenExpirationIntervalTimeUnit;
  }

  public Token grantAccessToken(final RequestGrantType grantType,
                                final AuthorizationRequest authorizationRequest,
                                final ResourceOwnerAuthentication resourceOwnerAuthentication) {
    checkArgument(authorizationRequest != null, "authorizationRequest can't be null");

    return grantAccessToken(grantType, authorizationRequest,
                            authorizationRequest.getClientId(), authorizationRequest.getScopes(), resourceOwnerAuthentication);
  }

  public Token grantAccessToken(final RequestGrantType grantType,
                                final String clientId,
                                final Set<String> scopes,
                                final ResourceOwnerAuthentication resourceOwnerAuthentication) {
    return grantAccessToken(grantType, null, clientId, scopes, resourceOwnerAuthentication);
  }

  private Token grantAccessToken(final RequestGrantType grantType,
                                 final AuthorizationRequest authorizationRequest,
                                 final String clientId,
                                 final Set<String> scopes,
                                 final ResourceOwnerAuthentication resourceOwnerAuthentication) {
    checkArgument(grantType != null, "grantType can't be null");
    checkArgument(!isEmpty(clientId), "clientId can't be empty");
    checkArgument(scopes != null, "scopes can't be null");

    final String refreshToken = refreshTokenStrategy.generateRefreshToken();
    final Token oauth2Token = new Token.Builder(clientId, tokenGeneratorStrategy.generateToken())
        .withRefreshToken(refreshToken)
        .withScopes(scopes)
        .withExpirationInterval(tokenExpirationInterval, tokenExpirationIntervalTimeUnit)
        .build();

    final ResourceOwnerAuthentication actualResourceOwnerAuthentication = resourceOwnerAuthentication == null
        && authorizationRequest != null
            ? authorizationRequest.getResourceOwnerAuthentication()
            : resourceOwnerAuthentication;

    tokenStore.store(new AccessTokenStoreHolder(oauth2Token, authorizationRequest, actualResourceOwnerAuthentication));

    return oauth2Token;
  }

  private Token refreshAccessToken(final AccessTokenStoreHolder previousHolder) throws InvalidGrantException {
    checkArgument(previousHolder != null, "previousHolder can't be null");

    if (tokenStore.remove(previousHolder.getAccessToken().getAccessToken()) == null) {
      throw new InvalidGrantException("Invalid refresh token");
    }

    final Token previousToken = previousHolder.getAccessToken();
    final String newAccessToken = tokenGeneratorStrategy.generateToken();
    final String newRefreshToken = refreshTokenStrategy.exchangeRefreshToken(previousToken.getRefreshToken());

    Token.Builder builder = new Token.Builder(newAccessToken, previousToken).withRefreshToken(newRefreshToken);
    final Token refreshedToken = builder.build();

    tokenStore.store(new AccessTokenStoreHolder(refreshedToken, previousHolder));

    return refreshedToken;
  }

  public AccessTokenStoreHolder getNonExpiredAccessTokenHolder(final String accessToken) {
    if (isBlank(accessToken)) {
      return null;
    }

    final AccessTokenStoreHolder holder = tokenStore.retrieveByAccessToken(accessToken);

    if (holder == null) {
      return null;
    }

    final Token token = holder.getAccessToken();
    if (isTokenExpired(accessToken)) {
      // Only remove the entry if the access token has no refresh token
      // or it does and it has expired.
      if (!token.hasRefreshToken() || isTokenExpired(token.getRefreshToken())) {
        tokenStore.remove(holder.getAccessToken().getAccessToken());
      }
      return null;
    }

    return holder;
  }

  public Token exchangeRefreshToken(final String refreshToken, final String clientId)
      throws InvalidGrantException {

    final AccessTokenStoreHolder holder = tokenStore.retrieveByRefreshToken(refreshToken);

    final Token currentAccessToken = holder == null ? null : holder.getAccessToken();

    if (currentAccessToken != null && isTokenExpired(currentAccessToken.getRefreshToken())) {
      tokenStore.removeByRefreshToken(refreshToken);
    }

    if ((currentAccessToken == null) || isTokenExpired(currentAccessToken.getRefreshToken())
        || (!currentAccessToken.getClientId().equals(clientId))) {
      throw new InvalidGrantException("Invalid or expired refresh token");
    }

    return refreshAccessToken(holder);
  }

  public boolean isTokenExpired(String token) {
    //Check if it is an access token
    AccessTokenStoreHolder tokenHolder = tokenStore.retrieveByAccessToken(token);

    if (tokenHolder != null) {
      //It's an existent access token but it may have an expired refresh token
      if (!tokenHolder.getAccessToken().hasRefreshToken()) {
        return false;
      }
      return isTokenExpired(tokenHolder.getAccessToken().getRefreshToken());
    }
    //It's not an access token, check if it's a refresh token
    tokenHolder = tokenStore.retrieveByRefreshToken(token);
    return tokenHolder == null;
  }

}
