/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal.token;

import static java.lang.String.format;
import static java.util.Collections.emptySet;
import static org.apache.commons.collections.CollectionUtils.intersection;
import static org.apache.commons.collections.CollectionUtils.isEmpty;
import static org.apache.commons.collections.CollectionUtils.isNotEmpty;
import org.mule.runtime.api.security.Authentication;
import org.mule.runtime.api.security.SecurityException;
import org.mule.runtime.core.api.security.AbstractSecurityProvider;

import com.mulesoft.modules.oauth2.provider.api.token.AccessTokenStoreHolder;
import com.mulesoft.modules.oauth2.provider.api.token.Token;
import com.mulesoft.modules.oauth2.provider.internal.OAuth2ProviderSecurityException;

import java.util.HashSet;
import java.util.Set;

/**
 * Security provider to authenticate
 */
public class TokenSecurityProvider extends AbstractSecurityProvider {

  public static final String NAME = "tokenSecurityProvider";

  private TokenManager tokenManager;

  public TokenSecurityProvider(String namePrefix, TokenManager tokenManager) {
    super(namePrefix + "." + NAME);
    this.tokenManager = tokenManager;
  }

  @Override
  public boolean supports(Class<?> aClass) {
    return TokenAuthentication.class.isAssignableFrom(aClass);
  }

  @Override
  public Authentication authenticate(Authentication authentication) throws SecurityException {
    if (!supports(authentication.getClass())) {
      throw new OAuth2ProviderSecurityException("Provider can't authenticate token class");
    }
    TokenAuthentication tokenAuthentication = (TokenAuthentication) authentication;

    String accessTokenToValidate = tokenAuthentication.getToken();
    AccessTokenStoreHolder storedTokenHolder = tokenManager.getNonExpiredAccessTokenHolder(accessTokenToValidate);

    //The token does not exist
    if (storedTokenHolder == null) {
      throw new UnauthorizedSecurityException(format("The token received: %s , is not valid", accessTokenToValidate));
    }

    Set<String> resourceOwnerRoles = tokenAuthentication.getResourceOwnerRoles();

    //Validate ResourceOwnerRoles
    if (isNotEmpty(resourceOwnerRoles)) {
      Set<String> savedResourceOwnerRoles = storedTokenHolder.getResourceOwnerAuthentication() == null ? emptySet()
          : storedTokenHolder.getResourceOwnerAuthentication().getRoles();
      if (isEmpty(intersection(resourceOwnerRoles, savedResourceOwnerRoles))) {
        throw new UnauthorizedSecurityException("Resource owner roles do not match");
      }
    }

    Set<String> scopesToValidate = tokenAuthentication.getScopes();

    //Validate scopes
    if (isNotEmpty(scopesToValidate)) {
      Set<String> savedScopesSet = new HashSet<>(storedTokenHolder.getAccessToken().getScopes());
      if (isEmpty(intersection(scopesToValidate, savedScopesSet))) {
        throw new ForbiddenSecurityException("Scopes do not match");
      }
    }

    Token savedToken = storedTokenHolder.getAccessToken();

    //Validate token string
    if (!accessTokenToValidate.equals(savedToken.getAccessToken())) {
      throw new UnauthorizedSecurityException("Tokens do not match");

    }

    //Validate token expiration time
    if (tokenManager.isTokenExpired(savedToken.getAccessToken())) {
      throw new UnauthorizedSecurityException("The token is expired");
    }

    return TokenAuthentication.builder(tokenAuthentication).withTokenHolder(storedTokenHolder).build();
  }

}
