/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.api.client;

import static com.mulesoft.modules.oauth2.provider.api.client.ClientType.CONFIDENTIAL;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.replace;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.core.api.util.StringUtils.isEmpty;

import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;

import com.mulesoft.modules.oauth2.provider.internal.processor.ClientSecretCredentials;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.regex.Pattern;

import javax.xml.namespace.QName;

import org.apache.commons.lang3.StringUtils;

/**
 * POJO for storing client's information.
 *
 * @since 1.0.0
 */
public final class Client implements Serializable, Component {

  private static final long serialVersionUID = -4691673785889272016L;

  //TODO:MULE-14997
  // Check if there is another fix for this. We have to declare the class as final and make it implement Component
  //to avoid enhancing it with CGLIB and make it serializable. If not, when trying to store it in an object store will
  //result in a NotSerializableException

  @Override
  public void setAnnotations(Map<QName, Object> annotations) {

  }

  @Override
  public Location getRootContainerLocation() {
    return null;
  }

  @Override
  public Object getAnnotation(QName name) {
    return null;
  }

  @Override
  public Map<QName, Object> getAnnotations() {
    return null;
  }

  @Override
  public ComponentLocation getLocation() {
    return null;
  }

  /**
   * A unique Id that will used to reference the client
   */
  @Parameter
  @Expression(SUPPORTED)
  private String clientId;

  /**
   * An optional principal to use when the ID can't be used with the security
   * provider.
   */
  @Parameter
  @Optional
  @Expression(SUPPORTED)
  private String principal;

  /**
   * The name of the client
   */
  @Parameter
  @Optional
  @Expression(SUPPORTED)
  private String clientName;

  /**
   * A short description for the client
   */
  @Parameter
  @Optional
  @Expression(SUPPORTED)
  private String description;

  /**
   * A client secret used to authenticate the client
   */
  @Parameter
  @Optional
  @Expression(SUPPORTED)
  private String secret;

  /**
   * List of registered redirectUris that will be used to respond to the client once the request has been processed.
   * </p>
   * Most request allow the definition of new redirection Uris so these will not always be taken into account.
   */
  @Parameter
  @Optional
  @NullSafe
  @Expression(SUPPORTED)
  private Set<String> clientRedirectUris;

  /**
   * The grant types that define which OAuth flows this client will be able to be successfully execute in order to get a token.
   */
  @Parameter
  @Optional
  @NullSafe
  @Expression(SUPPORTED)
  private Set<RequestGrantType> clientAuthorizedGrantTypes;

  /**
   * The scopes that will match this client. If a request is received with this clientId but different scopes, it will not be processed.
   */
  @Parameter
  @Optional
  @NullSafe
  @Expression(SUPPORTED)
  private Set<String> clientScopes;

  /**
   * The client type that defines if the client is able to maintain confidentiality of it's credentials.
   */
  @Parameter
  @Optional(defaultValue = "PUBLIC")
  private ClientType type;

  private transient ConcurrentMap<String, Pattern> redirectUriPatternCache;

  public Client() {
    redirectUriPatternCache = new ConcurrentHashMap<>();
  }

  public Client(final String clientId,
                final String secret,
                final ClientType type,
                final Set<String> clientRedirectUris,
                final Set<RequestGrantType> clientAuthorizedGrantTypes,
                final Set<String> clientScopes) {
    checkArgument(!isEmpty(clientId), "id can't be empty");
    this.clientId = clientId;
    this.secret = secret;
    this.type = type;
    this.clientRedirectUris = clientRedirectUris == null ? new HashSet<>() : clientRedirectUris;
    this.clientAuthorizedGrantTypes = clientAuthorizedGrantTypes == null ? new HashSet<>() : clientAuthorizedGrantTypes;
    this.clientScopes = clientScopes == null ? new HashSet<>() : clientScopes;
    redirectUriPatternCache = new ConcurrentHashMap<>();
  }

  public boolean isAuthenticatedBy(final ClientSecretCredentials clientSecretCredentials) {
    return isNotBlank(getSecret())
        && StringUtils.equals(getSecret(), clientSecretCredentials.getClientSecret());
  }

  public String getClientId() {
    return clientId;
  }

  public String getClientName() {
    return clientName;
  }

  public void setClientName(final String clientName) {
    this.clientName = clientName;
  }

  public String getDescription() {
    return description;
  }

  public void setDescription(final String description) {
    this.description = description;
  }

  public String getSecret() {
    return secret;
  }

  public void setSecret(final String secret) {
    this.secret = secret;
  }

  public Set<String> getRedirectUris() {
    return clientRedirectUris;
  }

  public void setRedirectUris(final Set<String> redirectUris) {
    this.clientRedirectUris.addAll(redirectUris);
  }

  public Set<RequestGrantType> getAuthorizedGrantTypes() {
    return clientAuthorizedGrantTypes;
  }

  public void setAuthorizedGrantTypes(final Set<RequestGrantType> authorizedGrantTypes) {
    this.clientAuthorizedGrantTypes.addAll(authorizedGrantTypes);
  }

  public ClientType getType() {
    return type;
  }

  public void setType(final ClientType type) {
    this.type = type;
  }

  public Set<String> getScopes() {
    return clientScopes;
  }

  public void setScopes(final Set<String> scopes) {
    this.clientScopes.addAll(scopes);
  }

  public String getPrincipal() {
    return principal;
  }

  public void setPrincipal(final String principal) {
    this.principal = principal;
  }

  public void setClientId(final String clientId) {
    this.clientId = clientId;
  }

  public boolean isValidRedirectUri(final String testedRedirectUri) {
    if (isBlank(testedRedirectUri)) {
      return false;
    }

    if (clientRedirectUris == null || clientRedirectUris.isEmpty()) {
      return false;
    }

    for (final String redirectUri : clientRedirectUris) {
      Pattern pattern = null;

      if (redirectUriPatternCache != null) {
        redirectUriPatternCache.get(redirectUri);
      }

      if (pattern == null) {
        // unquote and replace * with .*
        final String redirectUriRegex = replace(Pattern.quote(redirectUri), "*", "\\E.*\\Q");
        pattern = Pattern.compile(redirectUriRegex);
        redirectUriPatternCache.put(redirectUri, pattern);
      }

      if (pattern.matcher(testedRedirectUri).matches()) {
        return true;
      }
    }

    return false;
  }

  private void readObject(final ObjectInputStream s) throws ClassNotFoundException, IOException {
    s.defaultReadObject();
    redirectUriPatternCache = new ConcurrentHashMap<>();
  }

  public boolean isGrantTypeAuthorized(final RequestGrantType grantType) {
    return clientAuthorizedGrantTypes != null && clientAuthorizedGrantTypes.contains(grantType);
  }
}
