/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal;


import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.store.ObjectAlreadyExistsException;
import org.mule.runtime.api.store.ObjectDoesNotExistException;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreException;

import java.io.Serializable;

public abstract class AbstractObjectStoreBacked<T extends Serializable> {

  protected T getFromObjectStore(final String key, final ObjectStore<T> objectStore) {
    validateObjectStore(objectStore);

    if (key == null) {
      return null;
    }
    try {
      return objectStore.retrieve(key);
    } catch (final ObjectDoesNotExistException odnee) {
      return null;
    } catch (final ObjectStoreException ose) {
      // this is a serious problem so let's bubble it up
      throw new MuleRuntimeException(ose);
    }
  }

  protected void putInObjectStore(final String key, final T value, final ObjectStore<T> objectStore) {
    try {
      putInObjectStore(key, value, objectStore, false);
    } catch (ObjectAlreadyExistsException e) {
      throw new MuleRuntimeException(e);
    }
  }

  protected void putInObjectStore(final String key, final T value, final ObjectStore<T> objectStore, boolean failIfPresent)
      throws ObjectAlreadyExistsException {
    internalPutInObjectStore(key, value, objectStore, 3, failIfPresent);
  }

  private void internalPutInObjectStore(final String key,
                                        final T value,
                                        final ObjectStore<T> objectStore,
                                        final int attemptsLeft,
                                        final boolean failIfPresent)
      throws ObjectAlreadyExistsException {
    validateObjectStore(objectStore);
    try {
      // this is not atomic so can fail
      if (!failIfPresent && objectStore.contains(key)) {
        removeFromObjectStore(key, objectStore);
      }
      objectStore.store(key, value);
    } catch (final ObjectAlreadyExistsException oaee) {
      // the non atomic check-then-act above has failed
      if (attemptsLeft <= 0 || failIfPresent) {
        // bail
        throw oaee;
      }

      internalPutInObjectStore(key, value, objectStore, attemptsLeft - 1, failIfPresent);
    } catch (final ObjectStoreException ose) {
      // this is a serious problem so let's bubble it up
      throw new MuleRuntimeException(ose);
    }
  }

  protected T removeFromObjectStore(final String key, final ObjectStore<T> objectStore) {
    validateObjectStore(objectStore);
    try {
      return objectStore.remove(key);
    } catch (final ObjectDoesNotExistException odnee) {
      return null;
    } catch (final ObjectStoreException ose) {
      // this is a serious problem so let's bubble it up
      throw new MuleRuntimeException(ose);
    }
  }

  private void validateObjectStore(ObjectStore objectStore) {
    if (objectStore == null) {
      throw new IllegalArgumentException("No object store has been initialized.");
    }
  }
}
