/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal;

import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BEARER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ProviderGrantType.CLIENT_CREDENTIALS;
import static com.mulesoft.modules.oauth2.provider.api.client.ClientType.CONFIDENTIAL;
import static com.mulesoft.modules.oauth2.provider.api.client.ClientType.PUBLIC;
import static com.mulesoft.modules.oauth2.provider.api.client.ObjectStoreClientStore.CLIENTS_PARTITION;
import static com.mulesoft.modules.oauth2.provider.internal.Utils.parseProviderGrantTypes;
import static com.mulesoft.modules.oauth2.provider.internal.Utils.tokenize;
import static com.mulesoft.modules.oauth2.provider.api.code.ObjectStoreAuthorizationCode.AUTHORIZATION_CODE_PARTITION;
import static com.mulesoft.modules.oauth2.provider.api.token.ObjectStoreAccessAndRefreshTokenStore.ACCESS_TOKENS_PARTITION;
import static com.mulesoft.modules.oauth2.provider.api.token.ObjectStoreAccessAndRefreshTokenStore.REFRESH_TOKENS_PARTITION;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_STORE_MANAGER;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.util.ClassUtils.isClassOnPath;
import static org.mule.runtime.core.api.util.StringUtils.isBlank;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.ObjectStoreManager;
import org.mule.runtime.api.store.ObjectStoreSettings;
import org.mule.runtime.core.api.security.SecurityManager;
import org.mule.runtime.core.api.security.SecurityProvider;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Configuration;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.Operations;
import org.mule.runtime.extension.api.annotation.dsl.xml.ParameterDsl;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.RefName;
import org.mule.runtime.extension.api.annotation.param.reference.ConfigReference;
import org.mule.runtime.extension.api.annotation.param.reference.ObjectStoreReference;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.ServerNotFoundException;

import com.mulesoft.modules.oauth2.provider.api.Constants.ProviderGrantType;
import com.mulesoft.modules.oauth2.provider.api.client.Client;
import com.mulesoft.modules.oauth2.provider.api.client.ClientStore;
import com.mulesoft.modules.oauth2.provider.api.client.ObjectStoreClientStore;
import com.mulesoft.modules.oauth2.provider.api.code.AuthorizationConfig;
import com.mulesoft.modules.oauth2.provider.api.token.ObjectStoreAccessTokenStore;
import com.mulesoft.modules.oauth2.provider.api.token.TokenConfig;
import com.mulesoft.modules.oauth2.provider.internal.client.ClientManager;
import com.mulesoft.modules.oauth2.provider.internal.security.ResourceOwnerSecurityProvider;
import com.mulesoft.modules.oauth2.provider.internal.security.SpringAwareResourceOwnerSecurityProvider;
import com.mulesoft.modules.oauth2.provider.internal.token.generator.AbstractRefreshTokenStrategy;
import com.mulesoft.modules.oauth2.provider.internal.token.generator.ObjectStoreAwareRefreshTokenStrategy;
import com.mulesoft.modules.oauth2.provider.internal.code.AuthorizationCodeManager;
import com.mulesoft.modules.oauth2.provider.api.code.AuthorizationCodeStore;
import com.mulesoft.modules.oauth2.provider.api.code.ObjectStoreAuthorizationCode;
import com.mulesoft.modules.oauth2.provider.internal.config.IllegalConfigurationException;
import com.mulesoft.modules.oauth2.provider.internal.config.OAuthConfiguration;
import com.mulesoft.modules.oauth2.provider.internal.generator.AuthorizationHandlerGenerator;
import com.mulesoft.modules.oauth2.provider.internal.generator.CreateAccessTokenHandlerGenerator;
import com.mulesoft.modules.oauth2.provider.internal.generator.RequestHandlerGenerator;
import com.mulesoft.modules.oauth2.provider.api.ratelimit.RateLimiter;
import com.mulesoft.modules.oauth2.provider.api.ratelimit.PeriodRateLimiter;
import com.mulesoft.modules.oauth2.provider.api.token.ObjectStoreAccessAndRefreshTokenStore;
import com.mulesoft.modules.oauth2.provider.internal.token.TokenManager;
import com.mulesoft.modules.oauth2.provider.api.token.TokenStore;
import com.mulesoft.modules.oauth2.provider.internal.token.TokenSecurityProvider;
import com.mulesoft.modules.oauth2.provider.api.token.generator.TokenGeneratorDefaultStrategy;
import com.mulesoft.modules.oauth2.provider.api.token.generator.TokenGeneratorStrategy;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.inject.Inject;
import javax.inject.Named;

import org.slf4j.Logger;
import org.springframework.security.core.userdetails.UserDetailsService;

/**
 * OAuth2 Provider module configuration
 *
 * @since 1.0
 */
@Configuration
@Operations(OAuth2ProviderOperations.class)
public class OAuth2ProviderConfiguration implements Initialisable, Startable, Stoppable {

  private static final Logger logger = getLogger(OAuth2ProviderConfiguration.class);

  private static List<? extends RequestHandlerGenerator> REQUEST_HANDLER_GENERATORS;

  //Default OS config
  private static final int EXPIRATION_INTERVAL_PERCENTAGE = 10;
  private static final long DEFAULT_TOKEN_OS_ENTRY_TTL_MS = SECONDS.toMillis(86400);
  private static final long DEFAULT_AUTHORIZATION_CODE_OS_ENTRY_TTL_MS = SECONDS.toMillis(600);
  private static final boolean DEFAULT_PERSISTENCE_SETTING = true;

  private static final String DEFAULT_VALUES_DELIMITER = ",";


  public static String WWW_AUTHENTICATE_HEADER_VALUE = HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"";

  /**
   * The name of this configuration.
   */
  @RefName
  private String name;

  /**
   * The provider name supplied to customers of the API.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @Optional //The default value should be the same as the name.
  private String providerName;

  /**
   * Name of a valid listener configuration used to handle incoming requests.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @ConfigReference(namespace = "HTTP", name = "LISTENER_CONFIG")
  private String listenerConfig;


  /**
   * The rate limiter used to control access to certain operations. If none is specified {@link PeriodRateLimiter} will be
   * used as default.
   */
  @Parameter
  @Optional
  @NullSafe(defaultImplementingType = PeriodRateLimiter.class)
  private RateLimiter clientValidationRateLimiter;

  /**
   * A store that allows retrieving client configuration information, like their secret. If no client store is provided, a default
   * in memory object store is configured.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @Optional
  @Alias("clientStore")
  @ObjectStoreReference
  private ObjectStore clientStoreObjectStore;

  /**
   * The security provider used to authenticate resource owners. Not needed if only the CLIENT_CREDENTIALS grant type is used.
   */
  @Parameter
  @Optional
  @Alias("resourceOwnerSecurityProvider")
  private String resourceOwnerSecurityProvider;

  /**
   * The security provider used to authenticate clients. Not needed if only public clients or private clients with secrets are used
   */
  @Parameter
  @Optional
  @Alias("clientSecurityProvider")
  private String clientSecurityProvider;

  /**
   * The strategy used to generate access tokens. Should reference a class that implements {@link TokenGeneratorStrategy}
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @Optional
  @ParameterDsl(allowInlineDefinition = false)
  @NullSafe(defaultImplementingType = TokenGeneratorDefaultStrategy.class)
  private TokenGeneratorStrategy tokenGeneratorStrategy;

  /**
   * The comma-separated grant types this provider will support. If none specified, only the authorization code grant type will be
   * supported.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @Optional(defaultValue = "AUTHORIZATION_CODE")
  @Alias("supportedGrantTypes")
  private String supportedGrantTypesString;

  /**
   * A comma-separated list of supported scopes.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @Optional
  @Alias("scopes")
  private String scopesString;

  /**
   * A comma-separated list of the default scopes a client should have if none is defined.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @Optional
  @Alias("defaultScopes")
  private String defaultScopesString;

  /**
   * Information for configuring token related behaviour.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @Optional
  @NullSafe
  private TokenConfig tokenConfig;

  /**
   * Information for configuring authorization handling behaviour
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @Optional
  @NullSafe
  private AuthorizationConfig authorizationConfig;

  /**
   * A list of clients.
   */
  @Parameter
  @Optional
  @NullSafe
  private List<Client> clients;

  @Inject
  @Named(OBJECT_STORE_MANAGER)
  private ObjectStoreManager objectStoreManager;

  @Inject
  @Named("_muleSecurityManager")
  private SecurityManager securityManager;

  @Inject
  private HttpService httpService;

  @Inject
  private Registry registry;

  private OAuthConfiguration oAuthConfiguration;

  private TokenManager tokenManager;
  private TokenStore tokenStore;
  private TokenSecurityProvider tokenSecurityProvider;

  private ClientManager clientManager;
  private ClientStore clientStore;

  private AuthorizationCodeStore authorizationCodeStore;

  private Set<ProviderGrantType> supportedGrantTypes;
  private Set<String> scopes;
  private Set<String> defaultScopes;

  @Override
  public void initialise() {

    //The default value should be the same as the name
    if (providerName == null) {
      providerName = name;
    }

    supportedGrantTypes = parseProviderGrantTypes(supportedGrantTypesString);

    //Validate that a resourceOwnerSecurityProvider is set unless only CLIENT_CREDENTIALS is the only supported grant type.
    if (resourceOwnerSecurityProvider == null && isResourceOwnerSecurityProvidedNeeded()) {
      throw new IllegalConfigurationException(format("A Resource Owner Security Provided should be configured unless %s is the only supported grant type",
                                                     CLIENT_CREDENTIALS));
    }

    //Validate that a clientSecurityProvider is set unless only PUBLIC clients are used or CONFIDENTIAL with secrets set.
    if (clientSecurityProvider == null && isClientSecurityProviderNeeded()) {
      throw new IllegalConfigurationException(format("A Client Security Provided should be configured unless only %s clients are used or %s with secrets set",
                                                     PUBLIC,
                                                     CONFIDENTIAL));
    }

    scopes = scopesString != null ? tokenize(scopesString, DEFAULT_VALUES_DELIMITER) : new HashSet<>();
    defaultScopes = defaultScopesString != null ? tokenize(defaultScopesString, DEFAULT_VALUES_DELIMITER) : new HashSet<>();
    if (!scopes.containsAll(this.defaultScopes)) {
      throw new IllegalConfigurationException("Error configuring default scopes. Default scopes should be a subset of the configured scopes for the OAuth provider.");
    }
  }

  /**
   * Runs on initialise phase. Programatically generates and initialses the necessary flows to handle server svuide OAuth 2
   * authentication.
   *
   * @throws MuleException
   */
  @Override
  public void start() throws MuleException {
    REQUEST_HANDLER_GENERATORS =
        asList(new AuthorizationHandlerGenerator(), new CreateAccessTokenHandlerGenerator());
    HttpServer httpServer;
    try {
      httpServer = httpService.getServerFactory().lookup(listenerConfig);
    } catch (ServerNotFoundException e) {
      throw new MuleRuntimeException(createStaticMessage(format("OAuth provider '%s' defines '%s' as the http:listener-config to use for provisioning callbacks, but no such definition exists in the application configuration",
                                                                providerName, listenerConfig)),
                                     e);
    }
    final OAuthConfiguration OAuthConfiguration = createConfiguration(httpServer);
    for (final RequestHandlerGenerator requestHandlerGenerator : REQUEST_HANDLER_GENERATORS) {
      requestHandlerGenerator.generate(OAuthConfiguration);
    }

  }

  private OAuthConfiguration createConfiguration(HttpServer httpServer)
      throws MuleException {
    configureStores();

    clientManager = new ClientManager(clientStore);

    addClientsToStore();

    final AuthorizationCodeManager authorizationCodeManager = new AuthorizationCodeManager(authorizationCodeStore);
    ((AbstractRefreshTokenStrategy) tokenConfig.getRefreshTokenStrategy()).setTokenGeneratorStrategy(tokenGeneratorStrategy);
    tokenManager = new TokenManager(tokenStore,
                                    tokenGeneratorStrategy,
                                    tokenConfig.getRefreshTokenStrategy(),
                                    tokenConfig.getTokenTtl(),
                                    tokenConfig.getTokenTtlTimeUnit());

    tokenSecurityProvider = new TokenSecurityProvider(name, tokenManager);
    tokenSecurityProvider.initialise();

    securityManager.addProvider(tokenSecurityProvider);


    oAuthConfiguration =
        new OAuthConfiguration(providerName, httpServer,
                               createResourceOwnerSecurityProvider(),
                               createClientSecurityProvider(),
                               tokenConfig,
                               authorizationConfig,
                               clientManager, authorizationCodeManager,
                               tokenManager, scopes, defaultScopes,
                               supportedGrantTypes,
                               clientValidationRateLimiter);

    return oAuthConfiguration;
  }

  private boolean isResourceOwnerSecurityProvidedNeeded() {
    return (!supportedGrantTypes.contains(CLIENT_CREDENTIALS) || supportedGrantTypes.size() > 1);
  }

  private ResourceOwnerSecurityProvider createResourceOwnerSecurityProvider() {
    if (resourceOwnerSecurityProvider != null) {
      SecurityProvider foundResourceOwnerSecurityProvider = securityManager.getProvider(resourceOwnerSecurityProvider);
      if (foundResourceOwnerSecurityProvider == null) {
        throw new IllegalConfigurationException(format("Could not find resourceOwnerSecurityProvider referenced by the name: '%s'",
                                                       resourceOwnerSecurityProvider));
      }
      //Check if using spring or not
      if (isClassOnPath("org.springframework.security.core.userdetails.UserDetailsService",
                        getClass())) {
        Collection<UserDetailsService> userDetailsServices = registry.lookupAllByType(UserDetailsService.class);
        if (userDetailsServices == null) {
          userDetailsServices = emptyList();
        }
        return new SpringAwareResourceOwnerSecurityProvider(foundResourceOwnerSecurityProvider,
                                                            new ArrayList<>(userDetailsServices));
      } else {
        return new ResourceOwnerSecurityProvider(foundResourceOwnerSecurityProvider);
      }
    }
    return null;

  }

  private boolean isClientSecurityProviderNeeded() {
    //If there are only PUBLIC clients, we don't need the clientSecurityProvider.
    //Otherwise, if there is any CONFIDENTIAL client, we should check if they have a secret set. If that is the case
    //validation will be done without the use of a clientSecurityProvider (Just comparing with stored clients).
    return clients.stream().anyMatch(c -> c.getType().equals(CONFIDENTIAL) && isBlank(c.getSecret()));
  }

  private SecurityProvider createClientSecurityProvider() {
    if (clientSecurityProvider != null) {
      SecurityProvider foundClientSecurityProvider = securityManager.getProvider(clientSecurityProvider);
      if (foundClientSecurityProvider == null) {
        throw new IllegalConfigurationException(format("Could not find clientSecurityProvider referenced by the name: '%s'",
                                                       clientSecurityProvider));
      }
      return foundClientSecurityProvider;
    }
    return null;
  }

  @SuppressWarnings({"rawtypes", "unchecked"})
  private void configureStores() throws MuleException {
    initializeDefaultStores();
  }

  private void initializeDefaultStores() throws MuleException {
    initializeClientStore();
    initializeAuthorizationCodeStore();
    initializeTokenStore();
  }

  private void addClientsToStore() {
    if (clients != null) {
      for (final Client client : clients) {
        try {
          clientManager.addClient(client, false);
        } catch (IllegalArgumentException e) {
          throw new IllegalConfigurationException(e.getMessage());
        }
      }
    }
  }

  private void initializeClientStore() throws MuleException {
    clientStore = new ObjectStoreClientStore();
    if (useDefaultClientStore()) {
      ObjectStore<? extends Serializable> clientStoreObjectStore =
          objectStoreManager.getOrCreateObjectStore(CLIENTS_PARTITION,
                                                    ObjectStoreSettings
                                                        .builder()
                                                        .entryTtl(0L)
                                                        .expirationInterval(0L)
                                                        .persistent(DEFAULT_PERSISTENCE_SETTING)
                                                        .build());
      ((ObjectStoreClientStore) clientStore)
          .setObjectStore(clientStoreObjectStore);

      // For some reason the store directory is not solved unless explicitly initialized.
      try {
        ((ObjectStoreClientStore) this.clientStore).getClientObjectStore().open();
      } catch (ObjectStoreException ose) {
        throw new MuleRuntimeException(createStaticMessage("Error initializing persistent object store for Clients"));
      }
    } else {
      ((ObjectStoreClientStore) clientStore).setObjectStore(clientStoreObjectStore);
      startIfNeeded(clientStoreObjectStore);
    }
  }

  private void initializeAuthorizationCodeStore() throws MuleException {
    authorizationCodeStore = new ObjectStoreAuthorizationCode();
    if (useDefaultAuthorizationCodeStore()) {
      ObjectStore<? extends Serializable> authorizationCodeObjectStore =
          objectStoreManager.getOrCreateObjectStore(AUTHORIZATION_CODE_PARTITION,
                                                    ObjectStoreSettings
                                                        .builder()
                                                        .persistent(DEFAULT_PERSISTENCE_SETTING)
                                                        .entryTtl(DEFAULT_AUTHORIZATION_CODE_OS_ENTRY_TTL_MS)
                                                        .expirationInterval((DEFAULT_AUTHORIZATION_CODE_OS_ENTRY_TTL_MS
                                                            * EXPIRATION_INTERVAL_PERCENTAGE / 100))
                                                        .maxEntries(-1)
                                                        .build());
      ((ObjectStoreAuthorizationCode) authorizationCodeStore).setObjectStore(authorizationCodeObjectStore);
    } else {
      ((ObjectStoreAuthorizationCode) authorizationCodeStore)
          .setObjectStore(authorizationConfig.getAuthorizationCodeStore());
      startIfNeeded(authorizationConfig.getAuthorizationCodeStore());
    }
  }

  private void initializeTokenStore() throws MuleException {
    ObjectStoreAccessTokenStore tmpTokenStore;
    if (shouldStoreRefreshTokens()) {
      tmpTokenStore = new ObjectStoreAccessAndRefreshTokenStore();
      ObjectStore refreshTokensObjectStore =
          ((ObjectStoreAwareRefreshTokenStrategy) tokenConfig.getRefreshTokenStrategy()).getObjectStore();
      if (refreshTokensObjectStore != null) {
        startIfNeeded(refreshTokensObjectStore);
        tmpTokenStore.setRefreshTokenObjectStore(refreshTokensObjectStore);
      } else {
        tmpTokenStore.setRefreshTokenObjectStore(objectStoreManager.getOrCreateObjectStore(
                                                                                           REFRESH_TOKENS_PARTITION,
                                                                                           ObjectStoreSettings
                                                                                               .builder()
                                                                                               .persistent(DEFAULT_PERSISTENCE_SETTING)
                                                                                               .maxEntries(-1)
                                                                                               .entryTtl(DEFAULT_TOKEN_OS_ENTRY_TTL_MS)
                                                                                               .expirationInterval((DEFAULT_TOKEN_OS_ENTRY_TTL_MS
                                                                                                   * EXPIRATION_INTERVAL_PERCENTAGE
                                                                                                   / 100))
                                                                                               .build()));
      }
    } else {
      tmpTokenStore = new ObjectStoreAccessTokenStore();
    }

    if (useDefaultAccessTokenStore()) {
      final long tokenTtlMilliseconds = tokenConfig.getTokenTtlTimeUnit().toMillis(tokenConfig.getTokenTtl());
      tmpTokenStore.setAccessTokenObjectStore(objectStoreManager.getOrCreateObjectStore(
                                                                                        ACCESS_TOKENS_PARTITION,
                                                                                        ObjectStoreSettings
                                                                                            .builder()
                                                                                            .persistent(DEFAULT_PERSISTENCE_SETTING)
                                                                                            .maxEntries(-1)
                                                                                            .entryTtl(tokenTtlMilliseconds)
                                                                                            .expirationInterval((tokenTtlMilliseconds
                                                                                                * EXPIRATION_INTERVAL_PERCENTAGE
                                                                                                / 100))
                                                                                            .build()));
    } else {
      tmpTokenStore.setAccessTokenObjectStore(tokenConfig.getTokenStore());
      startIfNeeded(tokenConfig.getTokenStore());
    }
    tokenStore = tmpTokenStore;
  }

  private boolean useDefaultClientStore() {
    return clientStoreObjectStore == null;
  }

  private boolean useDefaultAccessTokenStore() {
    return tokenConfig == null || tokenConfig.getTokenStore() == null;
  }


  private boolean shouldStoreRefreshTokens() {
    return ObjectStoreAwareRefreshTokenStrategy.class.isAssignableFrom(tokenConfig.getRefreshTokenStrategy().getClass());
  }

  private boolean useDefaultAuthorizationCodeStore() {
    return authorizationConfig == null || authorizationConfig.getAuthorizationCodeStore() == null;
  }


  public OAuthConfiguration getOAuthConfiguration() {
    return oAuthConfiguration;
  }

  public ClientManager getClientManager() {
    return clientManager;
  }

  public AuthorizationCodeStore getAuthorizationCodeStore() {
    return authorizationCodeStore;
  }


  public TokenStore getTokenStore() {
    return tokenStore;
  }

  public TokenSecurityProvider getTokenSecurityProvider() {
    return tokenSecurityProvider;
  }


  @Override
  public void stop() throws MuleException {

    if (Objects.nonNull(securityManager.getProvider(tokenSecurityProvider.getName()))) {
      securityManager.removeProvider(tokenSecurityProvider.getName());
    }

    for (RequestHandlerGenerator generator : REQUEST_HANDLER_GENERATORS) {
      generator.dispose();
    }
  }
}
