/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal.code;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import com.mulesoft.modules.oauth2.provider.api.code.AuthorizationCodeStore;
import com.mulesoft.modules.oauth2.provider.api.code.AuthorizationCodeStoreHolder;
import com.mulesoft.modules.oauth2.provider.api.AuthorizationRequest;
import com.mulesoft.modules.oauth2.provider.api.exception.OAuth2Exception;
import com.mulesoft.modules.oauth2.provider.internal.Utils;
import com.mulesoft.modules.oauth2.provider.internal.token.InvalidGrantException;

public class AuthorizationCodeManager {

  private final AuthorizationCodeStore authorizationCodeStore;

  public AuthorizationCodeManager(final AuthorizationCodeStore authorizationCodeStore) {
    checkArgument(authorizationCodeStore != null, "authorizationCodeStore can't be null");

    this.authorizationCodeStore = authorizationCodeStore;
  }

  public String generateAuthorizationCode(final AuthorizationRequest authorizationRequest)
      throws OAuth2Exception {
    final String authorizationCode = Utils.generateUniqueId(authorizationRequest.getClientId());

    authorizationCodeStore.store(new AuthorizationCodeStoreHolder(authorizationCode, authorizationRequest));
    return authorizationCode;
  }

  public AuthorizationCodeStoreHolder consumeAuthorizationCode(final String code)
      throws InvalidGrantException {
    final AuthorizationCodeStoreHolder holder = isNotBlank(code)
        ? authorizationCodeStore.retrieve(code)
        : null;

    if (holder == null) {
      throw new InvalidGrantException("Authorization code is invalid or expired");
    }

    // Remove authorization code since it can only be consumed once
    authorizationCodeStore.remove(code);
    return holder;
  }
}
