/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal.config;

import static org.apache.commons.collections.CollectionUtils.isEmpty;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ProviderGrantType;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType;
import static com.mulesoft.modules.oauth2.provider.api.Constants.ResponseType;
import org.mule.runtime.core.api.security.SecurityProvider;
import org.mule.runtime.http.api.server.HttpServer;

import com.mulesoft.modules.oauth2.provider.api.code.AuthorizationConfig;
import com.mulesoft.modules.oauth2.provider.api.token.TokenConfig;
import com.mulesoft.modules.oauth2.provider.internal.client.ClientManager;
import com.mulesoft.modules.oauth2.provider.internal.code.AuthorizationCodeManager;
import com.mulesoft.modules.oauth2.provider.api.ratelimit.RateLimiter;
import com.mulesoft.modules.oauth2.provider.internal.security.ResourceOwnerSecurityProvider;
import com.mulesoft.modules.oauth2.provider.internal.token.TokenManager;

import java.util.Set;

public class OAuthConfiguration {

  private final String providerName;
  private final String host;
  private final int port;
  private final ResourceOwnerSecurityProvider resourceOwnerSecurityProvider;
  private final SecurityProvider clientSecurityProvider;
  private final TokenConfig tokenConfig;
  private final AuthorizationConfig authorizationConfig;
  private final ClientManager clientManager;
  private final TokenManager tokenManager;
  private final AuthorizationCodeManager authorizationCodeManager;
  private final Set<String> supportedScopes;
  private final Set<String> defaultScopes;
  private final Set<ProviderGrantType> supportedGrantTypes;
  private final RateLimiter rateLimiter;
  private final HttpServer httpServer;

  public OAuthConfiguration(final String providerName,
                            final HttpServer httpServer,
                            final ResourceOwnerSecurityProvider resourceOwnerSecurityProvider,
                            final SecurityProvider clientSecurityProvider,
                            final TokenConfig tokenConfig,
                            final AuthorizationConfig authorizationConfig,
                            final ClientManager clientManager,
                            final AuthorizationCodeManager authorizationCodeManager,
                            final TokenManager tokenManager,
                            final Set<String> scopes,
                            final Set<String> defaultScopes,
                            final Set<ProviderGrantType> supportedGrantTypes,
                            final RateLimiter rateLimiter) {
    this.providerName = providerName;
    this.httpServer = httpServer;
    this.host = httpServer.getServerAddress().getIp();
    this.port = httpServer.getServerAddress().getPort();
    this.resourceOwnerSecurityProvider = resourceOwnerSecurityProvider;
    this.clientSecurityProvider = clientSecurityProvider;
    this.tokenConfig = tokenConfig;
    this.authorizationConfig = authorizationConfig;
    this.clientManager = clientManager;
    this.tokenManager = tokenManager;
    this.authorizationCodeManager = authorizationCodeManager;
    this.supportedScopes = scopes;
    this.defaultScopes = defaultScopes;
    this.supportedGrantTypes = supportedGrantTypes;
    this.rateLimiter = rateLimiter;
  }

  public boolean isRequestGrantTypeSupported(final RequestGrantType requestGrantType) {
    checkArgument(requestGrantType != null, "requestGrantType can't be null");

    if (isEmpty(supportedGrantTypes)) {
      return false;
    }

    for (final ProviderGrantType providerGrantType : supportedGrantTypes) {
      if (providerGrantType.getRequestGrantTypes().contains(requestGrantType)) {
        return true;
      }
    }

    return false;
  }

  public boolean isAuthorizationResponseTypeSupported(final ResponseType authorizationResponseType) {
    checkArgument(authorizationResponseType != null, "authorizationResponseType can't be null");

    if (isEmpty(supportedGrantTypes)) {
      return false;
    }

    for (final ProviderGrantType providerGrantType : supportedGrantTypes) {
      if (providerGrantType.getAuthorizationResponseType() == authorizationResponseType) {
        return true;
      }
    }

    return false;
  }

  public String getProviderName() {
    return providerName;
  }

  public String getHost() {
    return host;
  }

  public int getPort() {
    return port;
  }

  public ResourceOwnerSecurityProvider getResourceOwnerSecurityProvider() {
    return resourceOwnerSecurityProvider;
  }

  public SecurityProvider getClientSecurityProvider() {
    return clientSecurityProvider;
  }

  public TokenConfig getTokenConfig() {
    return tokenConfig;
  }

  public AuthorizationConfig getAuthorizationConfig() {
    return authorizationConfig;
  }

  public ClientManager getClientManager() {
    return clientManager;
  }

  public TokenManager getTokenManager() {
    return tokenManager;
  }

  public AuthorizationCodeManager getAuthorizationCodeManager() {
    return authorizationCodeManager;
  }

  public Set<String> getSupportedScopes() {
    return supportedScopes;
  }

  public Set<String> getDefaultScopes() {
    return defaultScopes;
  }

  public Set<ProviderGrantType> getSupportedGrantTypes() {
    return supportedGrantTypes;
  }

  public RateLimiter getRateLimiter() {
    return rateLimiter;
  }

  public HttpServer getHttpServer() {
    return httpServer;
  }

}
